/*
 * Copyright 2025-present Solver4J
 *
 * This work is licensed under the Creative Commons Attribution-NoDerivatives 4.0 
 * International License. To view a copy of this license, visit 
 *
 *        http://creativecommons.org/licenses/by-nd/4.0/ 
 *
 * or send a letter to Creative Commons, PO Box 1866, Mountain View, CA 94042, USA.
 */
package com.solver4j.util;

import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.math3.linear.MatrixUtils;
import org.apache.commons.math3.linear.QRDecomposition;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.solver4j.util.Utils.BlockReduction;

import cern.colt.matrix.DoubleFactory1D;
import cern.colt.matrix.DoubleFactory2D;
import cern.colt.matrix.DoubleMatrix1D;
import cern.colt.matrix.DoubleMatrix2D;
import cern.colt.matrix.linalg.Algebra;

/**
 * @author <a href="mailto:orion.waverly@gmail.com">Orion Waverly</a>
 */
public class UtilsTest extends Solver4JBaseTest {

	private Algebra ALG = Algebra.DEFAULT;
	private DoubleFactory1D F1 = DoubleFactory1D.dense;
	private DoubleFactory2D F2 = DoubleFactory2D.dense;
	private Logger logger = LoggerFactory.getLogger(this.getClass().getName());

	public void testDummy() throws Exception{
		logger.debug("testDummy");
	}

	public void testNorm() throws Exception{
		logger.debug("testNorm");
		double[] d = new double[] { 1.1, 2.2, 3.3, 4.4, 5.5, 6.6, 7.7, 8.8, 9.9, 10.1 };
		DoubleMatrix1D v = F1.make(d);
		double n0 = v.zDotProduct(v);
		//double n1 = ALG.norm1(v);
		double n2 = ALG.norm2(v);
		logger.debug("n0: " + n0);
		//logger.debug("n1: " + n1);
		logger.debug("n2: " + n2);
		assertEquals(n0, n2, Utils.getDoubleMachineEpsilon());
	}
	
	public void testCalculateDeterminant() throws Exception{
		logger.debug("testCalculateDeterminant");
		double[][] A = new double[][]{{1, 0, 0},{2, 2, 0},{3, 3, 3}};
		double det = Utils.calculateDeterminant(A, 3);
		assertEquals(6.0, det);
	}
	
	public void testCalculateDeterminantBig() throws Exception{
		logger.debug("testCalculateDeterminantBig");
		DoubleMatrix2D A = ColtUtils.randomValuesSparseMatrix(40, 40, -1, 1, 0.9,	12345L);
		logger.debug("A: " + ArrayUtils.toString(A.toArray()));
		double det = Utils.calculateDeterminant(A.toArray(), A.rows());
		assertEquals(0., det, 1.e-6);
	}
	
	public void testQRdecomposition() throws Exception{
		logger.debug("testQRdecomposition");
		double[][] A = new double[][]{{1, 0, 0},{2, 2, 0},{3, 3, 3}};
		QRDecomposition qrFact = new QRDecomposition(MatrixUtils.createRealMatrix(A));
		qrFact.getQ();
	}
	
	public void testGetExpAndMantissa() throws Exception {
		logger.debug("testGetExpAndMantissa");
		float myFloat = -0.22f;
		Utils.getExpAndMantissa(myFloat);
		
		double myDouble = -0.22e-3;
		Utils.getExpAndMantissa(myDouble);
		
		myDouble = Math.pow(2, 17);
		Utils.getExpAndMantissa(myDouble);
	}
	
	public void testRound() throws Exception {
		logger.debug("testRound");
		double d = 0.1000000000000009;
		double precision = 1.e10;
		double d2 = Utils.round(d, precision);
		logger.debug("d2. " + d2);
		assertEquals(0.1, d2);
	}
	
	public void testAddToSortedArray() throws Exception {
		logger.debug("testAddToSortedArray");
		int[] array = new int[] { 0, 1, 2, 3, 4, 5 };

		int[] ret = Utils.addToSortedArray(array, 0);
		logger.debug("ret: " + ArrayUtils.toString(ret));
		assertTrue(ArrayUtils.isEquals(new int[] { 0, 1, 2, 3, 4, 5 }, ret));
		assertTrue(Utils.isInSortedArray(ret, 0));

		ret = Utils.addToSortedArray(array, 3);
		logger.debug("ret: " + ArrayUtils.toString(ret));
		assertTrue(ArrayUtils.isEquals(new int[] { 0, 1, 2, 3, 4, 5 }, ret));
		assertTrue(Utils.isInSortedArray(ret, 3));

		ret = Utils.addToSortedArray(array, 5);
		logger.debug("ret: " + ArrayUtils.toString(ret));
		assertTrue(ArrayUtils.isEquals(new int[] { 0, 1, 2, 3, 4, 5 }, ret));
		assertTrue(Utils.isInSortedArray(ret, 5));

		ret = Utils.addToSortedArray(array, 6);
		logger.debug("ret: " + ArrayUtils.toString(ret));
		assertTrue(ArrayUtils.isEquals(new int[] { 0, 1, 2, 3, 4, 5, 6 }, ret));
		assertTrue(Utils.isInSortedArray(ret, 6));

		ret = Utils.addToSortedArray(array, 8);
		logger.debug("ret: " + ArrayUtils.toString(ret));
		assertTrue(ArrayUtils.isEquals(new int[] { 0, 1, 2, 3, 4, 5, 8}, ret));
		assertTrue(Utils.isInSortedArray(ret, 8));

		ret = Utils.addToSortedArray(array, -1);
		logger.debug("ret: " + ArrayUtils.toString(ret));
		assertTrue(ArrayUtils.isEquals(new int[] { -1, 0, 1, 2, 3, 4, 5 }, ret));
		assertTrue(Utils.isInSortedArray(ret, -1));

		ret = ArrayUtils.remove(ret, 0);
		logger.debug("ret: " + ArrayUtils.toString(ret));
		assertTrue(ArrayUtils.isEquals(new int[] { 0, 1, 2, 3, 4, 5 }, ret));
		assertTrue(!Utils.isInSortedArray(ret, -1));

		ret = ArrayUtils.remove(ret, 0);
		logger.debug("ret: " + ArrayUtils.toString(ret));
		assertTrue(ArrayUtils.isEquals(new int[] { 1, 2, 3, 4, 5 }, ret));
		assertTrue(!Utils.isInSortedArray(ret, 0));

		ret = ArrayUtils.remove(ret, 0);
		logger.debug("ret: " + ArrayUtils.toString(ret));
		assertTrue(ArrayUtils.isEquals(new int[] { 2, 3, 4, 5 }, ret));
		assertTrue(!Utils.isInSortedArray(ret, 1));

		ret = ArrayUtils.remove(ret, 0);
		logger.debug("ret: " + ArrayUtils.toString(ret));
		assertTrue(ArrayUtils.isEquals(new int[] { 3, 4, 5 }, ret));
		assertTrue(!Utils.isInSortedArray(ret, 2));

		ret = ArrayUtils.remove(ret, 0);
		logger.debug("ret: " + ArrayUtils.toString(ret));
		assertTrue(ArrayUtils.isEquals(new int[] { 4, 5 }, ret));
		assertTrue(!Utils.isInSortedArray(ret, 3));

		ret = ArrayUtils.remove(ret, 0);
		logger.debug("ret: " + ArrayUtils.toString(ret));
		assertTrue(ArrayUtils.isEquals(new int[] { 5 }, ret));
		assertTrue(!Utils.isInSortedArray(ret, 4));

		ret = ArrayUtils.remove(ret, 0);
		logger.debug("ret: " + ArrayUtils.toString(ret));
		assertTrue(ArrayUtils.isEquals(new int[] {}, ret));
	}
	
	public void testBlockReduce1() throws Exception {
		logger.debug("testBlockReduce1");
		double[][] M = new double[][] { 
				{ 1, 1, 1, 0, 1, 0, 1, 0, 0, 0 },
				{ 0, 0, 1, 0, 1, 0, 0, 0, 0, 0 },
				{ 0, 0, 0, 1, 0, 1, 0, 0, 0, 0 },
				{ 0, 1, 1, 0, 0, 0, 0, 0, 0, 0 },
				{ 0, 0, 0, 0, 0, 0, 0, 1, 1, 1 }};
		//the connected components are: {{0,1,2,4,6},{7,8,9},{3,5}}
		
		BlockReduction br = Utils.blockReduce(M);
		double[][] MBlock = br.retMatrix;
		logger.debug("MBlock: " + ArrayUtils.toString(MBlock));
		int[][] Perm = br.createPermutationMatrix();
		logger.debug("Perm: " + ArrayUtils.toString(Perm));
		
		double[][] MBExpected = new double[][] {
				{ 1, 1, 1, 1, 1, 0, 0, 0, 0, 0 },
				{ 0, 0, 1, 1, 0, 0, 0, 0, 0, 0 },
				{ 0, 1, 1, 0, 0, 0, 0, 0, 0, 0 },
				{ 0, 0, 0, 0, 0, 1, 1, 1, 0, 0 },
				{ 0, 0, 0, 0, 0, 0, 0, 0, 1, 1 } };
		
		assertFalse(M == MBlock);
		for (int i = 0; i < M.length; i++) {
			for (int j = 0; j < M[0].length; j++) {
				assertTrue(Utils.isZero(MBExpected[i][j] - MBlock[i][j]));
			}
		}
	}
	
	public void xxxtestBlockReduce2() throws Exception {
		logger.debug("testBlockReduce2");
		double[][] M = new double[][] { 
				{ 1, 1, 1, 0, 0, 0, 0, 0, 0 },
				{ 0, 1, 1, 0, 0, 0, 0, 0, 0 },
				{ 0, 0, 0, 1, 0, 1, 0, 0, 0 },
				{ 0, 0, 0, 0, 0, 0, 1, 1, 1 },
				{ 0, 0, 1, 0, 1, 0, 0, 0, 0 } };
		//the connected components are: {{0,1,2,4},{6,7,8},{3,5}}
		
		//random permutation matrix
		//
		
		BlockReduction br = Utils.blockReduce(M);
		logger.debug("MBlock: " + ArrayUtils.toString(br.retMatrix));
		
		assertFalse(M == br.retMatrix);
	}
}
