/*
 * Copyright 2025-present Solver4J
 *
 * This work is licensed under the Creative Commons Attribution-NoDerivatives 4.0 
 * International License. To view a copy of this license, visit 
 *
 *        http://creativecommons.org/licenses/by-nd/4.0/ 
 *
 * or send a letter to Creative Commons, PO Box 1866, Mountain View, CA 94042, USA.
 */
package com.solver4j.util;

import java.io.File;
import java.util.Arrays;
import java.util.List;

import org.apache.commons.lang3.ArrayUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import junit.framework.TestCase;

/**
 * MPS parsing test.
 * 
 * @see "http://en.wikipedia.org/wiki/MPS_%28format%29"
 * @author <a href="mailto:orion.waverly@gmail.com">Orion Waverly</a>
 */
public class MPSParserLPTest extends Solver4JBaseTest {

	private Logger logger = LoggerFactory.getLogger(this.getClass().getName());

	/**
	 * This is the AFIRO netlib problem.
	 */
	public void testMps1() throws Exception {
		logger.debug("testMps1");
		
		String problemId = "1";
		
		File f = Utils.getClasspathResourceAsFile("lp" + File.separator	+ "mps" + File.separator + problemId + ".mps");
		double unspecifiedLBValue = 0;
		double unspecifiedUBValue = 99999;
		double unboundedLBValue = MPSParser.DEFAULT_UNBOUNDED_LOWER_BOUND;
		double unboundedUBValue = Double.POSITIVE_INFINITY;
		MPSParser p = new MPSParser(unspecifiedLBValue, unspecifiedUBValue, unboundedLBValue, unboundedUBValue);
		p.parse(f);
		
		int n = p.getN();
		int meq = p.getMeq();
		int mieq = p.getMieq();
		logger.debug("name: " + p.getName());
		logger.debug("n   : " + n);
		logger.debug("meq : " + meq);
		logger.debug("mieq: " + mieq);
		logger.debug("rows: " + (meq+mieq));
		logger.debug("nz: " + (p.getNzG() + p.getNzA()));
//		logger.debug("c   : " + ArrayUtils.toString(p.getC().toArray()));
//		logger.debug("G   : " + ArrayUtils.toString(p.getG().toArray()));
//		logger.debug("h   : " + ArrayUtils.toString(p.getH().toArray()));
//		logger.debug("A   : " + ArrayUtils.toString(p.getA().toArray()));
//		logger.debug("b   : " + ArrayUtils.toString(p.getB().toArray()));
		logger.debug("lb  : " + ArrayUtils.toString(p.getLb().toArray()));
		logger.debug("ub  : " + ArrayUtils.toString(p.getUb().toArray()));
		
		assertEquals(n, 32);
		assertEquals(meq, 8);
		assertEquals(mieq, 19);
		
		//all the lower bound are not explicit in this mps model
		for(int i=0; i<n; i++){
			assertEquals(unspecifiedLBValue, p.getLb().getQuick(i));
		}
		//all the upper bound are not explicit in this mps model
		for(int i=0; i<n; i++){
			assertEquals(unspecifiedUBValue, p.getUb().getQuick(i));
		}
	}
	
	/**
	 * This is the SCORPION netlib problem.
	 */
	public void testMps2() throws Exception {
		logger.debug("testMps2");
		
		String problemId = "2";
		
		File f = Utils.getClasspathResourceAsFile("lp" + File.separator	+ "mps" + File.separator + problemId + ".mps");
		MPSParser p = new MPSParser();
		p.parse(f);
		
		int n = p.getN();
		int meq = p.getMeq();
		int mieq = p.getMieq();
		logger.debug("name: " + p.getName());
		logger.debug("n   : " + n);
		logger.debug("meq : " + meq);
		logger.debug("mieq: " + mieq);
		logger.debug("rows: " + (meq+mieq));
		
		assertEquals(n, 358);
		assertEquals(meq, 280);
		assertEquals(mieq, 108);
	}
	
	/**
	 * This is the PILOT4 netlib problem.
	 */
	public void testMps3() throws Exception {
		logger.debug("testMps3");
		
		String problemId = "3";
		
		File f = Utils.getClasspathResourceAsFile("lp" + File.separator	+ "mps" + File.separator + problemId + ".mps");
		double unboundedLBValue = Double.NEGATIVE_INFINITY;
		double unboundedUBValue = Double.POSITIVE_INFINITY;
		double unspecifiedLBValue = 0;
		double unspecifiedUBValue = unboundedUBValue;
		MPSParser p = new MPSParser(unspecifiedLBValue, unspecifiedUBValue, unboundedLBValue, unboundedUBValue);
		p.parse(f);
		
		int n = p.getN();
		int meq = p.getMeq();
		int mieq = p.getMieq();
		logger.debug("name: " + p.getName());
		logger.debug("n   : " + n);
		logger.debug("meq : " + meq);
		logger.debug("mieq: " + mieq);
		logger.debug("rows: " + (meq+mieq));
		logger.debug("lb  : " + ArrayUtils.toString(p.getLb().toArray()));
		logger.debug("ub  : " + ArrayUtils.toString(p.getUb().toArray()));
		
		assertEquals(n, 1000);
		assertEquals(meq, 287);
		assertEquals(mieq, 123);
		
		List<String> unboundedVariables = Arrays.asList(new String[]{"XCRO01", "XROP01", "XGAS01", "XELE01", "XAGR01", "XMNG01", "XCMP01", "XFDS01", "XPPR01", "XSCG01", "XMET01", "XTEX01", "XLUM01", "XFAP01", "XMFG01", "XTAW01", "XTRD01", "XFIN01", "XSVC01", "XTRE01", "XMAC01"});
		List<String> variablesNames = p.getVariablesNames();
		for(int i=0; i<n; i++){
			String variable = variablesNames.get(i);
			if(unboundedVariables.contains(variable)){
				//this variables are stated to be unbounded in this mps model
				assertEquals(unboundedLBValue, p.getLb().getQuick(i));
				assertEquals(unboundedUBValue, p.getUb().getQuick(i));
			}else if("PLWU01".equalsIgnoreCase(variable)){
			  //this variables has not explicit bounds in this mps model
				assertEquals(unspecifiedLBValue, p.getLb().getQuick(i));
				assertEquals(unspecifiedUBValue, p.getUb().getQuick(i));
			}
		}
	}
	
	/**
	 * This is the STOCHFOR netlib problem.
	 */
	public void testMps4() throws Exception {
		logger.debug("testMps4");
		
		String problemId = "4";
		
		File f = Utils.getClasspathResourceAsFile("lp" + File.separator	+ "mps" + File.separator + problemId + ".mps");
		MPSParser p = new MPSParser();
		p.parse(f);
		
		int n = p.getN();
		int meq = p.getMeq();
		int mieq = p.getMieq();
		logger.debug("name: " + p.getName());
		logger.debug("n   : " + n);
		logger.debug("meq : " + meq);
		logger.debug("mieq: " + mieq);
		logger.debug("rows: " + (meq+mieq));
		
		assertEquals(n, 111);
		assertEquals(meq, 63);
		assertEquals(mieq, 54);
	}
	
}
