/*
 * Copyright 2025-present Solver4J
 *
 * This work is licensed under the Creative Commons Attribution-NoDerivatives 4.0 
 * International License. To view a copy of this license, visit 
 *
 *        http://creativecommons.org/licenses/by-nd/4.0/ 
 *
 * or send a letter to Creative Commons, PO Box 1866, Mountain View, CA 94042, USA.
 */
package com.solver4j.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.math3.linear.Array2DRowRealMatrix;
import org.apache.commons.math3.linear.ArrayRealVector;
import org.apache.commons.math3.linear.RealMatrix;
import org.apache.commons.math3.linear.RealVector;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * MPS parsing test.
 * 
 * @see "http://en.wikipedia.org/wiki/MPS_%28format%29"
 * @author <a href="mailto:orion.waverly@gmail.com">Orion Waverly</a>
 */
public class MPSParserLPNetlibTest extends Solver4JBaseTest {

	private Logger logger = LoggerFactory.getLogger(this.getClass().getName());

	/**
	 * Tests the parsing of a netlib problem.
	 */
	public void xxxtestSingleNetlib() throws Exception {
		logger.debug("testSingleNetlib");
		//String problemName = "afiro";
		//String problemName = "afiroPresolved";
		//String problemName = "adlittle";
		//String problemName = "kb2";
		//String problemName = "sc50a";
		//String problemName = "sc50b";
		//String problemName = "blend";
		//String problemName = "scorpion";
		//String problemName = "recipe";
		//String problemName = "recipePresolved";
		//String problemName = "sctap1";
		//String problemName = "fit1d";
		//String problemName = "israel";
		//String problemName = "grow15";
		//String problemName = "etamacro";
		//String problemName = "pilot";
		//String problemName = "pilot4";
		//String problemName = "osa-14";
		//String problemName = "brandyPresolved";
		String problemName = "maros";
		
		File f = Utils.getClasspathResourceAsFile("lp" + File.separator	+ "netlib" + File.separator + problemName + File.separator	+ problemName + ".mps");
		MPSParser mpsParser = new MPSParser();
		mpsParser.parse(f);
		
		Properties expectedSolProps = null;
		try{
			//this is the solution of the mps problem given by Mathematica
			expectedSolProps = load(Utils.getClasspathResourceAsFile("lp" + File.separator	+ "netlib" + File.separator + problemName + File.separator	+ "sol.txt"));
		}catch(Exception e){}

		logger.debug("name: " + mpsParser.getName());
		logger.debug("n   : " + mpsParser.getN());
		logger.debug("meq : " + mpsParser.getMeq());
		logger.debug("mieq: " + mpsParser.getMieq());
		logger.debug("meq+mieq: " + (mpsParser.getMeq()+mpsParser.getMieq()));
		List<String> variablesNames = mpsParser.getVariablesNames();
		logger.debug("x: " + ArrayUtils.toString(variablesNames));		
//		logger.debug("c: " + ArrayUtils.toString(p.getC()));
//		logger.debug("G: " + ArrayUtils.toString(p.getG()));
//		logger.debug("h: " + ArrayUtils.toString(p.getH()));
//		logger.debug("A: " + ArrayUtils.toString(p.getA()));
//		logger.debug("b: " + ArrayUtils.toString(p.getB()));
//		logger.debug("lb:" + ArrayUtils.toString(p.getLb()));
//		logger.debug("ub:" + ArrayUtils.toString(p.getUb()));
		
		//check consistency: if the problem was correctly parsed, the expectedSol must be its solution
		double delta = 1.e-7;
		if(expectedSolProps!=null){
			//key = variable name
			//value = sol value
			assertEquals(expectedSolProps.size(), variablesNames.size());
			RealVector expectedSol = new ArrayRealVector(variablesNames.size());
			for(int i=0; i<variablesNames.size(); i++){
				expectedSol.setEntry(i, Double.parseDouble(expectedSolProps.getProperty(variablesNames.get(i))));
			}
			logger.debug("expectedSol: " + ArrayUtils.toString(expectedSol.toArray()));
			
			//check objective function value
			Map<String, LPNetlibProblem> problemsMap = LPNetlibProblem.loadAllProblems();
			LPNetlibProblem problem = problemsMap.get(problemName);
			RealVector c = new ArrayRealVector(mpsParser.getC().toArray());
			double value = c.dotProduct(expectedSol);
			logger.debug("optimalValue: " + problem.optimalValue);
			logger.debug("value       : " + value);
			assertEquals(problem.optimalValue, value, delta);
			
			//check G.x < h
			if(mpsParser.getG()!=null){
				RealMatrix G = new Array2DRowRealMatrix(mpsParser.getG().toArray());
				RealVector h = new ArrayRealVector(mpsParser.getH().toArray());
				RealVector Gxh = G.operate(expectedSol).subtract(h);
				double maxGxh = -Double.MAX_VALUE;
				for(int i=0; i<Gxh.getDimension(); i++){
					//logger.debug(i);
					maxGxh = Math.max(maxGxh, Gxh.getEntry(i));
					assertTrue(Gxh.getEntry(i) <= 0);
				}
				logger.debug("max(G.x - h): " + maxGxh);
			}
			
			//check A.x = b
			if(mpsParser.getA()!=null){
				RealMatrix A = new Array2DRowRealMatrix(mpsParser.getA().toArray());
				RealVector b = new ArrayRealVector(mpsParser.getB().toArray());
				RealVector Axb = A.operate(expectedSol).subtract(b);
				double norm = Axb.getNorm();
				logger.debug("||A.x -b||: " + norm);
				assertEquals(0., norm, delta * mpsParser.getN());//some more tolerance
			}
			
			//check upper and lower bounds
			for(int i=0; i<mpsParser.getLb().size(); i++){
				double di = Double.isNaN(mpsParser.getLb().getQuick(i))? -Double.MAX_VALUE : mpsParser.getLb().getQuick(i);
				assertTrue(di <= expectedSol.getEntry(i));
			}
			for(int i=0; i<mpsParser.getUb().size(); i++){
				double di = Double.isNaN(mpsParser.getUb().getQuick(i))? Double.MAX_VALUE : mpsParser.getUb().getQuick(i);
				assertTrue(di >= expectedSol.getEntry(i));
			}
		}
		
		super.writeDoubleArrayToFile(mpsParser.getC().toArray(), "target" + File.separator	+ "c.txt");
		super.writeDoubleMatrixToFile(mpsParser.getG().toArray(), "target" + File.separator	+ "G.csv");
		super.writeDoubleArrayToFile(mpsParser.getH().toArray(), "target" + File.separator	+ "h.txt");
		if(1==1){
			super.writeDoubleMatrixToFile(mpsParser.getA().toArray(), "target" + File.separator	+ "A.csv");
			super.writeDoubleArrayToFile(mpsParser.getB().toArray(), "target" + File.separator	+ "b.txt");
		}
	} 
	
	/**
	 * Tests the parsing of all the netlib problems.
	 */
	public void testBulkNetlib() throws Exception {
		logger.debug("testBulkNetlib");
		Map<String, LPNetlibProblem> problemsMap = LPNetlibProblem.loadAllProblems();
		//List<NetlibLPResult> list = NetlibLPResult.getProblemsOrderedByName(resultsMap);
		//List<NetlibLPResult> list = NetlibLPResult.getProblemsOrderedByNumberOfRows(resultsMap);
		List<LPNetlibProblem> list = LPNetlibProblem.getProblemsOrderedByNumberOfColumns(problemsMap);
		for(LPNetlibProblem problem : list){
			logger.debug("problem: " + problem);
			//System.out.println(res.name);
			File f = Utils.getClasspathResourceAsFile("lp" + File.separator	+ "netlib" + File.separator + problem.name + File.separator	+ problem.name + ".mps");
			MPSParser p = new MPSParser();
			p.parse(f);
		}
	}
	
	public Properties load(File propsFile) throws IOException {
	    Properties props = new Properties();
	    FileInputStream fis = new FileInputStream(propsFile);
	    props.load(fis);    
	    fis.close();
	    return props;
	}
}
