/*
 * Copyright 2025-present Solver4J
 *
 * This work is licensed under the Creative Commons Attribution-NoDerivatives 4.0 
 * International License. To view a copy of this license, visit 
 *
 *        http://creativecommons.org/licenses/by-nd/4.0/ 
 *
 * or send a letter to Creative Commons, PO Box 1866, Mountain View, CA 94042, USA.
 */
package com.solver4j.util;

import java.io.BufferedInputStream;
import java.io.DataInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;


/**
 * Represent a Netlib BIP problem as contained in the bundle from
 * http://miplib.zib.de/
 * 
 * @author <a href="mailto:orion.waverly@gmail.com">Orion Waverly</a>
 */
public class BIPNetlibProblem {

	public String name;
	public int nz;
	public int rows;
	public int columns;
	public double optimalValue;
	
	public static Map<String, BIPNetlibProblem> loadAllProblems() throws Exception{
		File f = Utils.getClasspathResourceAsFile("ip" + File.separator	+ "bip" + File.separator + "netlib" + File.separator + "problemsList.csv");
		DataInputStream in = new DataInputStream(new BufferedInputStream(new FileInputStream(f)));
		Map<String, BIPNetlibProblem> problemsMap = new HashMap<String, BIPNetlibProblem>();
		String line = new String();
		//Name;NZ;Rows;Columns;Solution
		while ((line = in.readLine()) != null){
			if(line.startsWith("#")){
				//this is a comment
				continue;
			}
			BIPNetlibProblem problem = new BIPNetlibProblem();
			StringTokenizer st = new StringTokenizer(line, ";");
			problem.name = st.nextToken();
			//System.out.println(result.name);
			problem.nz = Integer.parseInt(st.nextToken());
			String strRows = st.nextToken();
			problem.rows = ("".equals(strRows))? 0 : Integer.parseInt(strRows);
			String strCols = st.nextToken();
			problem.columns = ("".equals(strCols))? 0 : Integer.parseInt(strCols);
			problem.optimalValue = Double.parseDouble(st.nextToken()); 
			problemsMap.put(problem.name, problem);
		}
		return problemsMap;
	}
	
	@Override
	public String toString(){
		return "name: " + name + ", nz: " + nz + ", rows: " + rows + ", cols: " + columns + ", optimalValue: " +optimalValue;
	}
	
	public static List<BIPNetlibProblem> getProblemsOrderedByName(Map<String, BIPNetlibProblem> resultsMap){
		List<BIPNetlibProblem> orderedList = new ArrayList<BIPNetlibProblem>();
		orderedList.addAll(resultsMap.values());
		Collections.sort(orderedList, new Comparator<BIPNetlibProblem>() {
			public int compare(BIPNetlibProblem arg0, BIPNetlibProblem arg1) {
				return arg0.name.compareTo(arg1.name);
			}
		});
		return orderedList;
	}
	
	public static List<BIPNetlibProblem> getProblemsOrderedByNumberOfRows(Map<String, BIPNetlibProblem> resultsMap){
		List<BIPNetlibProblem> orderedList = new ArrayList<BIPNetlibProblem>();
		orderedList.addAll(resultsMap.values());
		Collections.sort(orderedList, new Comparator<BIPNetlibProblem>() {
			public int compare(BIPNetlibProblem arg0, BIPNetlibProblem arg1) {
				if(arg0.rows  > arg1.rows){
					return 1;
				}else if(arg0.rows  < arg1.rows){
					return -1;
				}else{
					return 0;
				}
			}
		});
		return orderedList;
	}
	
	public static List<BIPNetlibProblem> getProblemsOrderedByNumberOfColumns(Map<String, BIPNetlibProblem> resultsMap){
		List<BIPNetlibProblem> orderedList = new ArrayList<BIPNetlibProblem>();
		orderedList.addAll(resultsMap.values());
		Collections.sort(orderedList, new Comparator<BIPNetlibProblem>() {
			public int compare(BIPNetlibProblem arg0, BIPNetlibProblem arg1) {
				if(arg0.columns  > arg1.columns){
					return 1;
				}else if(arg0.columns  < arg1.columns){
					return -1;
				}else{
					return 0;
				}
			}
		});
		return orderedList;
	}
	
	public static List<BIPNetlibProblem> getProblemsOrderedBySize(Map<String, BIPNetlibProblem> resultsMap){
		List<BIPNetlibProblem> orderedList = new ArrayList<BIPNetlibProblem>();
		orderedList.addAll(resultsMap.values());
		Collections.sort(orderedList, new Comparator<BIPNetlibProblem>() {
			public int compare(BIPNetlibProblem arg0, BIPNetlibProblem arg1) {
				if(arg0.rows*arg0.columns  > arg1.rows*arg1.columns){
					return 1;
				}else if(arg0.rows*arg0.columns  < arg1.rows*arg1.columns){
					return -1;
				}else{
					return 0;
				}
			}
		});
		return orderedList;
	}
}
