/*
 * Copyright 2025-present Solver4J
 *
 * This work is licensed under the Creative Commons Attribution-NoDerivatives 4.0 
 * International License. To view a copy of this license, visit 
 *
 *        http://creativecommons.org/licenses/by-nd/4.0/ 
 *
 * or send a letter to Creative Commons, PO Box 1866, Mountain View, CA 94042, USA.
 */
package com.solver4j.solvers;

import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.math3.linear.Array2DRowRealMatrix;
import org.apache.commons.math3.linear.ArrayRealVector;
import org.apache.commons.math3.linear.CholeskyDecomposition;
import org.apache.commons.math3.linear.RealMatrix;
import org.apache.commons.math3.linear.RealVector;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.solver4j.functions.PDQuadraticMultivariateRealFunction;
import com.solver4j.linear.factorization.CholeskyFactorization;
import com.solver4j.util.ColtUtils;
import com.solver4j.util.Solver4JBaseTest;

import cern.colt.matrix.DoubleFactory1D;
import cern.colt.matrix.DoubleFactory2D;
import cern.colt.matrix.DoubleMatrix1D;
import cern.colt.matrix.DoubleMatrix2D;
import cern.colt.matrix.linalg.Algebra;
import cern.jet.math.Functions;

/**
 * @author <a href="mailto:orion.waverly@gmail.com">Orion Waverly</a>
 */
public class NewtonUnconstrainedTest extends Solver4JBaseTest {
	private Algebra ALG = Algebra.DEFAULT;
	private DoubleFactory1D F1 = DoubleFactory1D.dense;
	private DoubleFactory2D F2 = DoubleFactory2D.dense;
	private Logger logger = LoggerFactory.getLogger(this.getClass().getName());

	/**
	 * Quadratic objective.
	 */
	public void testOptimize() throws Exception {
		logger.debug("testOptimize");
		// START SNIPPET: newtonUnconstrained-1
		
		RealMatrix PMatrix = new Array2DRowRealMatrix(new double[][] {
				{ 1.68, 0.34, 0.38 },
				{ 0.34, 3.09, -1.59 }, 
				{ 0.38, -1.59, 1.54 } });
		RealVector qVector = new ArrayRealVector(new double[] { 0.018, 0.025, 0.01 });

	    // Objective function.
		double theta = 0.01522;
		RealMatrix P = PMatrix.scalarMultiply(theta);
		RealVector q = qVector.mapMultiply(-1);
		PDQuadraticMultivariateRealFunction objectiveFunction = new PDQuadraticMultivariateRealFunction(P.getData(), q.toArray(), 0);
		
		OptimizationRequest or = new OptimizationRequest();
		or.setF0(objectiveFunction);
		or.setInitialPoint(new double[] {0.04, 0.50, 0.46});
		or.setTolerance(1.e-8);
		
	    //optimization
		NewtonUnconstrained opt = new NewtonUnconstrained();
		opt.setOptimizationRequest(or);
		opt.optimize(); 
		
		// END SNIPPET: newtonUnconstrained-1
		
		OptimizationResponse response = opt.getOptimizationResponse();
		double[] sol = response.getSolution();
		logger.debug("sol   : " + ArrayUtils.toString(sol));
		logger.debug("value : "	+ objectiveFunction.value(F1.make(sol)));

		// we know the analytic solution of the problem
		// sol = -PInv * q
		CholeskyDecomposition cFact = new CholeskyDecomposition(P);
		RealVector benchSol = cFact.getSolver().solve(q).mapMultiply(-1);
		logger.debug("benchSol   : " + ArrayUtils.toString(benchSol.toArray()));
		logger.debug("benchValue : " + objectiveFunction.value(F1.make(benchSol.toArray())));

		assertEquals(benchSol.getEntry(0), sol[0], 0.00000000000001);
		assertEquals(benchSol.getEntry(1), sol[1], 0.00000000000001);
		assertEquals(benchSol.getEntry(2), sol[2], 0.00000000000001);
	}

	/**
	 * Test with quite large positive definite symmetric matrix.
	 */
	public void testOptimize2() throws Exception {
		logger.debug("testOptimize2");

		int dim = 40;
		
		// positive definite matrix
		Long seed = new Long(54321);
		DoubleMatrix2D mySymmPD = ColtUtils.randomValuesPositiveMatrix(dim, dim, -0.01, 15.5, seed);
		DoubleMatrix1D CVector = ColtUtils.randomValuesMatrix(1, dim, -0.01, 15.5, seed).viewRow(0);
		MySymmFunction objectiveFunction = new MySymmFunction(mySymmPD,	CVector);
		
		//optimization
		OptimizationRequest or = new OptimizationRequest();
		or.setF0(objectiveFunction);
		NewtonUnconstrained opt = new NewtonUnconstrained();
		opt.setOptimizationRequest(or);
		opt.optimize();
		
		OptimizationResponse response = opt.getOptimizationResponse();
		double[] sol = response.getSolution();
		logger.debug("sol   : " + ArrayUtils.toString(sol));
		logger.debug("value : " + objectiveFunction.value(F1.make(sol)));
		
		// we know the analytic solution of the problem: Qinv.sol = - C
		CholeskyFactorization cFact = new CholeskyFactorization(mySymmPD);
		cFact.factorize();
		DoubleMatrix2D B = F2.make(CVector.copy().assign(Functions.mult(-1)).toArray(), (int) CVector.size());
		DoubleMatrix1D benchSol = cFact.solve(B).viewColumn(0);
		logger.debug("benchSol   : " + ArrayUtils.toString(benchSol.toArray()));
		logger.debug("benchValue : "	+ objectiveFunction.value(F1.make(benchSol.toArray())));

		for(int i=0; i<dim;i++){
			assertEquals(benchSol.get(i), sol[i], 0.000001);
		}
	}

	private class MySymmFunction extends PDQuadraticMultivariateRealFunction {

		public MySymmFunction(DoubleMatrix2D P, DoubleMatrix1D q) {
			super(P, q, 0);
		}

	}
}
