/*
 * Copyright 2025-present Solver4J
 *
 * This work is licensed under the Creative Commons Attribution-NoDerivatives 4.0 
 * International License. To view a copy of this license, visit 
 *
 *        http://creativecommons.org/licenses/by-nd/4.0/ 
 *
 * or send a letter to Creative Commons, PO Box 1866, Mountain View, CA 94042, USA.
 */
package com.solver4j.solvers;

import java.io.File;

import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.math3.linear.Array2DRowRealMatrix;
import org.apache.commons.math3.linear.ArrayRealVector;
import org.apache.commons.math3.linear.MatrixUtils;
import org.apache.commons.math3.linear.RealMatrix;
import org.apache.commons.math3.linear.RealVector;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.solver4j.util.Solver4JBaseTest;

/**
 * Standard form conversion test.
 * 
 * @author <a href="mailto:orion.waverly@gmail.com">Orion Waverly</a>
 */
public class LPStandardConverterTest extends Solver4JBaseTest {

	private Logger logger = LoggerFactory.getLogger(this.getClass().getName());
	
	/**
	 * Standardization of a problem on the form:
	 * min(c) s.t.
	 * G.x < h
	 * A.x = b
	 * lb <= x <= ub
	 */
	public void testCGhAbLbUb1() throws Exception {
		logger.debug("testCGhAbLbUb1");
		
		String problemId = "1";
		
		double[] c = super.loadDoubleArrayFromFile("lp"+File.separator+"standardization"+File.separator+"c"+problemId+".txt");
		double[][] G = super.loadDoubleMatrixFromFile("lp"+File.separator+"standardization"+File.separator+"G"+problemId+".csv", ",".charAt(0));
		double[] h = super.loadDoubleArrayFromFile("lp"+File.separator+"standardization"+File.separator+"h"+problemId+".txt");;
		double[][] A = super.loadDoubleMatrixFromFile("lp"+File.separator+"standardization"+File.separator+"A"+problemId+".csv", ",".charAt(0));
		double[] b = super.loadDoubleArrayFromFile("lp"+File.separator+"standardization"+File.separator+"b"+problemId+".txt");
		double[] lb = super.loadDoubleArrayFromFile("lp"+File.separator+"standardization"+File.separator+"lb"+problemId+".txt");
		double[] ub = super.loadDoubleArrayFromFile("lp"+File.separator+"standardization"+File.separator+"ub"+problemId+".txt");
		double[] expectedSol = super.loadDoubleArrayFromFile("lp"+File.separator+"standardization"+File.separator+"sol"+problemId+".txt");
		double expectedValue = super.loadDoubleArrayFromFile("lp"+File.separator+"standardization"+File.separator+"value"+problemId+".txt")[0];
		//double expectedTolerance = super.loadDoubleArrayFromFile("lp"+File.separator+"standardization"+File.separator+"tolerance"+problemId+".txt")[0];
		double expectedTolerance = MatrixUtils.createRealMatrix(A).operate(MatrixUtils.createRealVector(expectedSol)).subtract(MatrixUtils.createRealVector(b)).getNorm(); 
			
		//standard form conversion
		double unboundedLBValue = Double.NEGATIVE_INFINITY;//this is because in the file the unbounded lb are -Infinity values (not the default value) 
		double unboundedUBValue = Double.POSITIVE_INFINITY;//this is because in the file the unbounded ub are +Infinity values
		LPStandardConverter lpConverter = new LPStandardConverter(unboundedLBValue, unboundedUBValue);
		lpConverter.toStandardForm(c, G, h, A, b, lb, ub);
		
		int n = lpConverter.getStandardN();
		int s = lpConverter.getStandardS();
		c = lpConverter.getStandardC().toArray();
		A = lpConverter.getStandardA().toArray();
		b = lpConverter.getStandardB().toArray();
		lb = lpConverter.getStandardLB().toArray();
		ub = lpConverter.getStandardUB().toArray();
		logger.debug("n : " + n);
		logger.debug("s : " + s);
		logger.debug("c : " + ArrayUtils.toString(c));
		logger.debug("A : " + ArrayUtils.toString(A));
		logger.debug("b : " + ArrayUtils.toString(b));
		logger.debug("lb : " + ArrayUtils.toString(lb));
		logger.debug("ub : " + ArrayUtils.toString(ub));
		
	  //check consistency
		assertEquals(G.length, s);
		assertEquals(s + lpConverter.getOriginalN(), n);
		assertEquals(lb.length, n);
		assertEquals(ub.length, n);
		
	//check constraints
		RealMatrix GOrig = new Array2DRowRealMatrix(G);
		RealVector hOrig = new ArrayRealVector(h);
		RealMatrix AStandard = new Array2DRowRealMatrix(A);
		RealVector bStandard = new ArrayRealVector(b);
		RealVector expectedSolVector = new ArrayRealVector(expectedSol);
		RealVector Gxh = GOrig.operate(expectedSolVector).subtract(hOrig);//G.x - h
		RealVector slackVariables = new ArrayRealVector(s);
		for(int i=0; i<s; i++){
			slackVariables.setEntry(i, 0. - Gxh.getEntry(i));//the difference from 0
			assertTrue(slackVariables.getEntry(i) >= 0.);
		}
		RealVector sol = slackVariables.append(expectedSolVector);
		RealVector Axmb = AStandard.operate(sol).subtract(bStandard);
		assertEquals(0., Axmb.getNorm(), expectedTolerance);
		
//		super.writeDoubleArrayToFile(new double[]{s}, "target" + File.separator	+ "standardS"+problemId+".txt");
//		super.writeDoubleArrayToFile(c, "target" + File.separator	+ "standardC"+problemId+".txt");
//		super.writeDoubleMatrixToFile(A, "target" + File.separator	+ "standardA"+problemId+".txt");
//		super.writeDoubleArrayToFile(b, "target" + File.separator	+ "standardB"+problemId+".txt");
//		super.writeDoubleArrayToFile(lb, "target" + File.separator	+ "standardLB"+problemId+".txt");
//		super.writeDoubleArrayToFile(ub, "target" + File.separator	+ "standardUB"+problemId+".txt");
	}
	
	/**
	 * Standardization (to the strictly standard form) of a problem on the form:
	 * min(c) s.t.
	 * G.x < h
	 * A.x = b
	 * lb <= x <= ub
	 * TODO: the strict conversion is net yet ready.
	 */
	public void xxxtestCGhAbLbUb1Strict() throws Exception {
		logger.debug("testCGhAbLbUb1Strict");
		
		String problemId = "1";
		
		double[] c = super.loadDoubleArrayFromFile("lp"+File.separator+"standardization"+File.separator+"c"+problemId+".txt");
		double[][] G = super.loadDoubleMatrixFromFile("lp"+File.separator+"standardization"+File.separator+"G"+problemId+".csv", ",".charAt(0));
		double[] h = super.loadDoubleArrayFromFile("lp"+File.separator+"standardization"+File.separator+"h"+problemId+".txt");;
		double[][] A = super.loadDoubleMatrixFromFile("lp"+File.separator+"standardization"+File.separator+"A"+problemId+".csv", ",".charAt(0));
		double[] b = super.loadDoubleArrayFromFile("lp"+File.separator+"standardization"+File.separator+"b"+problemId+".txt");
		double[] lb = super.loadDoubleArrayFromFile("lp"+File.separator+"standardization"+File.separator+"lb"+problemId+".txt");
		double[] ub = super.loadDoubleArrayFromFile("lp"+File.separator+"standardization"+File.separator+"ub"+problemId+".txt");
		double[] expectedSol = super.loadDoubleArrayFromFile("lp"+File.separator+"standardization"+File.separator+"sol"+problemId+".txt");
		double expectedValue = super.loadDoubleArrayFromFile("lp"+File.separator+"standardization"+File.separator+"value"+problemId+".txt")[0];
		double expectedTolerance = MatrixUtils.createRealMatrix(A).operate(MatrixUtils.createRealVector(expectedSol)).subtract(MatrixUtils.createRealVector(b)).getNorm();
		
		int nOfSlackVariables = 0;
		for(int i=0; i<c.length; i++){
			double lbi = lb[i];
			int lbCompare = Double.compare(lbi, 0.); 
			if(lbCompare != 0 && !Double.isNaN(lbi)){
				nOfSlackVariables++;
			}
			if(!Double.isNaN(ub[i])){
				nOfSlackVariables++;
			}
		}
		int expectedS = G.length + nOfSlackVariables;
		
		//standard form conversion
		boolean strictlyStandardForm = true;
		LPStandardConverter lpConverter = new LPStandardConverter(strictlyStandardForm);
		lpConverter.toStandardForm(c, G, h, A, b, lb, ub);
		
		int n = lpConverter.getStandardN();
		int s = lpConverter.getStandardS();
		c = lpConverter.getStandardC().toArray();
		A = lpConverter.getStandardA().toArray();
		b = lpConverter.getStandardB().toArray();
		lb = lpConverter.getStandardLB().toArray();
		ub = (lpConverter.getStandardUB()==null)? null : ub;
		logger.debug("n : " + n);
		logger.debug("s : " + s);
		logger.debug("c : " + ArrayUtils.toString(c));
		logger.debug("A : " + ArrayUtils.toString(A));
		logger.debug("b : " + ArrayUtils.toString(b));
		logger.debug("lb : " + ArrayUtils.toString(lb));
		//logger.debug("ub : " + ArrayUtils.toString(ub));
		
		//check consistency
		assertEquals(expectedS, s);
		assertEquals(lb.length, n);
		assertTrue(ub == null);
		
		//check constraints
		RealMatrix GOrig = new Array2DRowRealMatrix(G);
		RealVector hOrig = new ArrayRealVector(h);
		RealMatrix AStandard = new Array2DRowRealMatrix(A);
		RealVector bStandard = new ArrayRealVector(b);
		RealVector expectedSolVector = new ArrayRealVector(expectedSol);
		double[] expectedStandardSol = lpConverter.getStandardComponents(expectedSol);
		RealVector expectedStandardSolVector = new ArrayRealVector(expectedStandardSol);
		
		for(int i=0; i<expectedStandardSolVector.getDimension(); i++){
			assertTrue(expectedStandardSolVector.getEntry(i) >= 0.);
		}
		
		RealVector Axmb = AStandard.operate(expectedStandardSolVector).subtract(bStandard);
		assertEquals(0., Axmb.getNorm(), expectedTolerance);
		
		super.writeDoubleArrayToFile(new double[]{s}, "target" + File.separator	+ "standardS_"+problemId+".txt");
		super.writeDoubleArrayToFile(c, "target" + File.separator	+ "standardC_"+problemId+".txt");
		super.writeDoubleMatrixToFile(A, "target" + File.separator	+ "standardA_"+problemId+".csv");
		super.writeDoubleArrayToFile(b, "target" + File.separator	+ "standardB_"+problemId+".txt");
		super.writeDoubleArrayToFile(lb, "target" + File.separator	+ "standardLB_"+problemId+".txt");
		//ub is null super.writeDoubleArrayToFile(ub, "target" + File.separator	+ "standardUB_"+problemId+".txt");
	}
	
	/**
	 * Standardization of a problem on the form:
	 * min(c) s.t.
	 * G.x < h
	 * A.x = b
	 */
	public void testCGhAb2() throws Exception {
		logger.debug("testCGhAb2");
		
		String problemId = "2";
		
		double[] c = super.loadDoubleArrayFromFile("lp"+File.separator+"standardization"+File.separator+"c"+problemId+".txt");
		double[][] G = super.loadDoubleMatrixFromFile("lp"+File.separator+"standardization"+File.separator+"G"+problemId+".csv", ",".charAt(0));
		double[] h = super.loadDoubleArrayFromFile("lp"+File.separator+"standardization"+File.separator+"h"+problemId+".txt");;
		double[][] A = super.loadDoubleMatrixFromFile("lp"+File.separator+"standardization"+File.separator+"A"+problemId+".csv", ",".charAt(0));
		double[] b = super.loadDoubleArrayFromFile("lp"+File.separator+"standardization"+File.separator+"b"+problemId+".txt");
		double[] expectedSol = super.loadDoubleArrayFromFile("lp"+File.separator+"standardization"+File.separator+"sol"+problemId+".txt");
		double expectedValue = super.loadDoubleArrayFromFile("lp"+File.separator+"standardization"+File.separator+"value"+problemId+".txt")[0];
		double expectedTolerance = MatrixUtils.createRealMatrix(A).operate(MatrixUtils.createRealVector(expectedSol)).subtract(MatrixUtils.createRealVector(b)).getNorm();
		
		//standard form conversion
		double unboundedLBValue = Double.NEGATIVE_INFINITY;
		double unboundedUBValue = Double.POSITIVE_INFINITY;
		LPStandardConverter lpConverter = new LPStandardConverter(unboundedLBValue, unboundedUBValue);
		lpConverter.toStandardForm(c, G, h, A, b, null, null);
		
		int n = lpConverter.getStandardN();
		int s = lpConverter.getStandardS();
		c = lpConverter.getStandardC().toArray();
		A = lpConverter.getStandardA().toArray();
		b = lpConverter.getStandardB().toArray();
		double[] lb = lpConverter.getStandardLB().toArray();
		double[] ub = lpConverter.getStandardUB().toArray();
		logger.debug("n : " + n);
		logger.debug("s : " + s);
		logger.debug("c : " + ArrayUtils.toString(c));
		logger.debug("A : " + ArrayUtils.toString(A));
		logger.debug("b : " + ArrayUtils.toString(b));
		logger.debug("lb : " + ArrayUtils.toString(lb));
		logger.debug("ub : " + ArrayUtils.toString(ub));
		
	  //check consistency
		assertEquals(G.length, s);
		assertEquals(A[0].length, n);
		assertEquals(s + lpConverter.getOriginalN(), n);
		assertEquals(lb.length, n);
		assertEquals(ub.length, n);
		
		//check constraints
		RealMatrix GOrig = new Array2DRowRealMatrix(G);
		RealVector hOrig = new ArrayRealVector(h);
		RealMatrix AStandard = new Array2DRowRealMatrix(A);
		RealVector bStandard = new ArrayRealVector(b);
		RealVector expectedSolVector = new ArrayRealVector(expectedSol);
		RealVector Gxh = GOrig.operate(expectedSolVector).subtract(hOrig);//G.x - h
		RealVector slackVariables = new ArrayRealVector(s);
		for(int i=0; i<s; i++){
			slackVariables.setEntry(i, 0. - Gxh.getEntry(i));//the difference from 0
			assertTrue(slackVariables.getEntry(i) >= 0.);
		}
		RealVector sol = slackVariables.append(expectedSolVector);
		RealVector Axmb = AStandard.operate(sol).subtract(bStandard);
		assertEquals(0., Axmb.getNorm(), expectedTolerance);
		
//		super.writeDoubleArrayToFile(new double[]{s}, "target" + File.separator	+ "standardS"+problemId+".txt");
//		super.writeDoubleArrayToFile(c, "target" + File.separator	+ "standardC"+problemId+".txt");
//		super.writeDoubleMatrixToFile(A, "target" + File.separator	+ "standardA"+problemId+".txt");
//		super.writeDoubleArrayToFile(b, "target" + File.separator	+ "standardB"+problemId+".txt");
//		super.writeDoubleArrayToFile(lb, "target" + File.separator	+ "standardLB"+problemId+".txt");
//		super.writeDoubleArrayToFile(ub, "target" + File.separator	+ "standardUB"+problemId+".txt");
	}
	
	/**
	 * Standardization of a problem on the form:
	 * min(c) s.t.
	 * G.x < h
	 * A.x = b
	 */
	public void testCGhAb3() throws Exception {
		logger.debug("testCGhAb3");
		
		String problemId = "3";
		
		double[] c = super.loadDoubleArrayFromFile("lp"+File.separator+"standardization"+File.separator+"c"+problemId+".txt");
		double[][] G = super.loadDoubleMatrixFromFile("lp"+File.separator+"standardization"+File.separator+"G"+problemId+".csv", ",".charAt(0));
		double[] h = super.loadDoubleArrayFromFile("lp"+File.separator+"standardization"+File.separator+"h"+problemId+".txt");;
		double[][] A = super.loadDoubleMatrixFromFile("lp"+File.separator+"standardization"+File.separator+"A"+problemId+".csv", ",".charAt(0));
		double[] b = super.loadDoubleArrayFromFile("lp"+File.separator+"standardization"+File.separator+"b"+problemId+".txt");
		double[] expectedSol = super.loadDoubleArrayFromFile("lp"+File.separator+"standardization"+File.separator+"sol"+problemId+".txt");
		double expectedValue = super.loadDoubleArrayFromFile("lp"+File.separator+"standardization"+File.separator+"value"+problemId+".txt")[0];
		double expectedTolerance = MatrixUtils.createRealMatrix(A).operate(MatrixUtils.createRealVector(expectedSol)).subtract(MatrixUtils.createRealVector(b)).getNorm();
		
		//standard form conversion
		LPStandardConverter lpConverter = new LPStandardConverter();
		lpConverter.toStandardForm(c, G, h, A, b, null, null);
		
		int n = lpConverter.getStandardN();
		int s = lpConverter.getStandardS();
		c = lpConverter.getStandardC().toArray();
		A = lpConverter.getStandardA().toArray();
		b = lpConverter.getStandardB().toArray();
		double[] lb = lpConverter.getStandardLB().toArray();
		double[] ub = lpConverter.getStandardUB().toArray();
		logger.debug("n : " + n);
		logger.debug("s : " + s);
		
		//check consistency
		assertEquals(G.length, s);
		assertEquals(A[0].length, n);
		assertEquals(s + lpConverter.getOriginalN(), n);
		assertEquals(lb.length, n);
		assertEquals(ub.length, n);
		
	//check constraints
		RealMatrix GOrig = new Array2DRowRealMatrix(G);
		RealVector hOrig = new ArrayRealVector(h);
		RealMatrix AStandard = new Array2DRowRealMatrix(A);
		RealVector bStandard = new ArrayRealVector(b);
		RealVector expectedSolVector = new ArrayRealVector(expectedSol);
		RealVector Gxh = GOrig.operate(expectedSolVector).subtract(hOrig);//G.x - h
		RealVector slackVariables = new ArrayRealVector(s);
		for(int i=0; i<s; i++){
			slackVariables.setEntry(i, 0. - Gxh.getEntry(i));//the difference from 0
			assertTrue(slackVariables.getEntry(i) >= 0.);
		}
		RealVector sol = slackVariables.append(expectedSolVector);
		RealVector Axmb = AStandard.operate(sol).subtract(bStandard);
		assertEquals(0., Axmb.getNorm(), expectedTolerance);
		
//		super.writeDoubleArrayToFile(new double[]{s}, "target" + File.separator	+ "standardS"+problemId+".txt");
//		super.writeDoubleArrayToFile(c, "target" + File.separator	+ "standardC"+problemId+".txt");
//		super.writeDoubleMatrixToFile(A, "target" + File.separator	+ "standardA"+problemId+".txt");
//		super.writeDoubleArrayToFile(b, "target" + File.separator	+ "standardB"+problemId+".txt");
//		super.writeDoubleArrayToFile(lb, "target" + File.separator	+ "standardLB"+problemId+".txt");
//		super.writeDoubleArrayToFile(ub, "target" + File.separator	+ "standardUB"+problemId+".txt");
	}

	/**
	 * Standardization of a problem on the form:
	 * min(c) s.t.
	 * G.x < h
	 * A.x = b
	 * lb <= x <= ub
	 */
	public void testCGhAbLbUb4() throws Exception {
		logger.debug("testCGhAbLbUb4");
		
		String problemId = "4";
		
		double[] c = super.loadDoubleArrayFromFile("lp"+File.separator+"standardization"+File.separator+"c"+problemId+".txt");
		double[][] G = super.loadDoubleMatrixFromFile("lp"+File.separator+"standardization"+File.separator+"G"+problemId+".csv", ",".charAt(0));
		double[] h = super.loadDoubleArrayFromFile("lp"+File.separator+"standardization"+File.separator+"h"+problemId+".txt");;
		double[][] A = super.loadDoubleMatrixFromFile("lp"+File.separator+"standardization"+File.separator+"A"+problemId+".csv", ",".charAt(0));
		double[] b = super.loadDoubleArrayFromFile("lp"+File.separator+"standardization"+File.separator+"b"+problemId+".txt");
		double[] lb = super.loadDoubleArrayFromFile("lp"+File.separator+"standardization"+File.separator+"lb"+problemId+".txt");
		double[] ub = super.loadDoubleArrayFromFile("lp"+File.separator+"standardization"+File.separator+"ub"+problemId+".txt");
		double[] expectedSol = super.loadDoubleArrayFromFile("lp"+File.separator+"standardization"+File.separator+"sol"+problemId+".txt");
		double expectedValue = super.loadDoubleArrayFromFile("lp"+File.separator+"standardization"+File.separator+"value"+problemId+".txt")[0];
		double expectedTolerance = MatrixUtils.createRealMatrix(A).operate(MatrixUtils.createRealVector(expectedSol)).subtract(MatrixUtils.createRealVector(b)).getNorm();
		
		int nOsSplittingVariables = 0;
//		for(int i=0; i<lb.length; i++){
//			if(Double.compare(lb[i], 0.) != 0){
//				nOsSplittingVariables++;
//			}
//		}
		
		//standard form conversion
		double unboundedLBValue = Double.NaN;//this is because in the file the unbounded lb are NaN values (and also the default value) 
		double unboundedUBValue = Double.NaN;//this is because in the file the unbounded ub are NaN values
		LPStandardConverter lpConverter = new LPStandardConverter(unboundedLBValue, unboundedUBValue);
		lpConverter.toStandardForm(c, G, h, A, b, lb, ub);
		
		int n = lpConverter.getStandardN();
		int s = lpConverter.getStandardS();
		c = lpConverter.getStandardC().toArray();
		A = lpConverter.getStandardA().toArray();
		b = lpConverter.getStandardB().toArray();
		lb = lpConverter.getStandardLB().toArray();
		ub = lpConverter.getStandardUB().toArray();
		logger.debug("n : " + n);
		logger.debug("s : " + s);
		
	  //check consistency
		assertEquals(G.length, s);
		assertEquals(s + lpConverter.getOriginalN() + nOsSplittingVariables, n);
		assertEquals(lb.length, n);
		assertEquals(ub.length, n);
		
	  //check constraints
		RealMatrix GOrig = new Array2DRowRealMatrix(G);
		RealVector hOrig = new ArrayRealVector(h);
		RealMatrix AStandard = new Array2DRowRealMatrix(A);
		RealVector bStandard = new ArrayRealVector(b);
		RealVector expectedSolVector = new ArrayRealVector(expectedSol);
		RealVector Gxh = GOrig.operate(expectedSolVector).subtract(hOrig);//G.x - h
		RealVector slackVariables = new ArrayRealVector(s);
		for(int i=0; i<s; i++){
			slackVariables.setEntry(i, 0. - Gxh.getEntry(i));//the difference from 0
			assertTrue(slackVariables.getEntry(i) >= 0.);
		}
		RealVector sol = slackVariables.append(expectedSolVector);
		RealVector Axmb = AStandard.operate(sol).subtract(bStandard);
		assertEquals(0., Axmb.getNorm(), expectedTolerance * 1.001);
		
//		super.writeDoubleArrayToFile(new double[]{s}, "target" + File.separator	+ "standardS"+problemId+".txt");
//		super.writeDoubleArrayToFile(c, "target" + File.separator	+ "standardC"+problemId+".txt");
//		super.writeDoubleMatrixToFile(A, "target" + File.separator	+ "standardA"+problemId+".txt");
//		super.writeDoubleArrayToFile(b, "target" + File.separator	+ "standardB"+problemId+".txt");
//		super.writeDoubleArrayToFile(lb, "target" + File.separator	+ "standardLB"+problemId+".txt");
//		super.writeDoubleArrayToFile(ub, "target" + File.separator	+ "standardUB"+problemId+".txt");
	}
	
	/**
	 * Standardization of a problem on the form:
	 * min(c) s.t.
	 * G.x < h
	 * A.x = b
	 * lb <= x <= ub
	 */
	public void testCGhAbLbUb5() throws Exception {
		logger.debug("testCGhAbLbUb5");
		
		double[] c = new double[]{1.0,1.0,1.0,1.0,1.0};
		double[][] G = new double[][] { { 0.0, -1.0, -1.0, -1.0, -1.0 } };
		double[] h = new double[] { -0.20127217432317202 };
		double[][] A = new double[][] { 
			{ 1.0, 1.0, 1.0, 1.0, 1.0 }, 
			{ 0.0, 1.0, 1.0, 1.0, 1.0 }, 
			{ 1.0, 0.0, 0.0, 0.0, 0.0 } };
		double[] b = new double[] { 1.0, 0.23481753671036731, 0.7651824632896327 };
		double[] lb = new double[] { 0.7651824632896327, 0.0, 0.0, 0.0, 0.0 };
		double[] ub = new double[] { 0.7651824632896327, 0.05870438417759183, 0.05870438417759183, 0.05870438417759183, 0.05870438417759183 };
		double[] expectedSol = new double[]{0.7651824632896327, 0.05870438417759183, 0.05870438417759183, 0.05870438417759183, 0.05870438417759183};
		double expectedValue = 1d;
		double expectedTolerance = MatrixUtils.createRealMatrix(A).operate(MatrixUtils.createRealVector(expectedSol)).subtract(MatrixUtils.createRealVector(b)).getNorm();
		
		int nOsSplittingVariables = 0;
		
		//standard form conversion
		double unboundedLBValue = Double.NaN;//this is because in the file the unbounded lb are NaN values (and also the default value) 
		double unboundedUBValue = Double.NaN;//this is because in the file the unbounded ub are NaN values
		LPStandardConverter lpConverter = new LPStandardConverter(unboundedLBValue, unboundedUBValue);
		lpConverter.toStandardForm(c, G, h, A, b, lb, ub);
		
		int n = lpConverter.getStandardN();
		int s = lpConverter.getStandardS();
		c = lpConverter.getStandardC().toArray();
		A = lpConverter.getStandardA().toArray();
		b = lpConverter.getStandardB().toArray();
		lb = lpConverter.getStandardLB().toArray();
		ub = lpConverter.getStandardUB().toArray();
		logger.debug("n : " + n);
		logger.debug("s : " + s);
		
	    //check consistency
		assertEquals(G.length, s);
		assertEquals(s + lpConverter.getOriginalN() + nOsSplittingVariables, n);
		assertEquals(lb.length, n);
		assertEquals(ub.length, n);
		
	    //check constraints
		RealMatrix GOrig = new Array2DRowRealMatrix(G);
		RealVector hOrig = new ArrayRealVector(h);
		RealMatrix AStandard = new Array2DRowRealMatrix(A);
		RealVector bStandard = new ArrayRealVector(b);
		RealVector expectedSolVector = new ArrayRealVector(expectedSol);
		RealVector Gxh = GOrig.operate(expectedSolVector).subtract(hOrig);//G.x - h
		RealVector slackVariables = new ArrayRealVector(s);
		for(int i=0; i<s; i++){
			slackVariables.setEntry(i, 0. - Gxh.getEntry(i));//the difference from 0
			assertTrue(slackVariables.getEntry(i) >= 0.);
		}
		RealVector sol = slackVariables.append(expectedSolVector);
		RealVector Axmb = AStandard.operate(sol).subtract(bStandard);
		assertEquals(0., Axmb.getNorm(), expectedTolerance * 1.001);
		
//		super.writeDoubleArrayToFile(new double[]{s}, "target" + File.separator	+ "standardS"+problemId+".txt");
//		super.writeDoubleArrayToFile(c, "target" + File.separator	+ "standardC"+problemId+".txt");
//		super.writeDoubleMatrixToFile(A, "target" + File.separator	+ "standardA"+problemId+".txt");
//		super.writeDoubleArrayToFile(b, "target" + File.separator	+ "standardB"+problemId+".txt");
//		super.writeDoubleArrayToFile(lb, "target" + File.separator	+ "standardLB"+problemId+".txt");
//		super.writeDoubleArrayToFile(ub, "target" + File.separator	+ "standardUB"+problemId+".txt");
	}
	
	/**
	 * Standardization (to the strictly standard form) of a problem on the form:
	 * min(c) s.t.
	 * A.x = b
	 * lb <= x <= ub
	 * 
	 * This is the presolved (with Solver4J) pilot4 netlib problem.
	 * TODO: the strict conversion is net yet ready.
	 */
	public void xxxtestCAbLbUb5Strict() throws Exception {
		logger.debug("testCAbLbUb5Strict");
		
		String problemId = "5";
		
		double[] c = super.loadDoubleArrayFromFile("lp"+File.separator+"standardization"+File.separator+"c"+problemId+".txt");
		double[][] A = super.loadDoubleMatrixFromFile("lp"+File.separator+"standardization"+File.separator+"A"+problemId+".csv", ",".charAt(0));
		double[] b = super.loadDoubleArrayFromFile("lp"+File.separator+"standardization"+File.separator+"b"+problemId+".txt");
		double[] lb = super.loadDoubleArrayFromFile("lp"+File.separator+"standardization"+File.separator+"lb"+problemId+".txt");
		double[] ub = super.loadDoubleArrayFromFile("lp"+File.separator+"standardization"+File.separator+"ub"+problemId+".txt");
		double[] expectedSol = super.loadDoubleArrayFromFile("lp"+File.separator+"standardization"+File.separator+"sol"+problemId+".txt");
		double expectedValue = super.loadDoubleArrayFromFile("lp"+File.separator+"standardization"+File.separator+"value"+problemId+".txt")[0];
		double expectedTol = super.loadDoubleArrayFromFile("lp"+File.separator+"standardization"+File.separator+"tolerance"+problemId+".txt")[0];
		
		int nOfSlackVariables = 0;
		for(int i=0; i<c.length; i++){
			double lbi = lb[i];
			int lbCompare = Double.compare(lbi, 0.); 
			if(lbCompare != 0 && !Double.isNaN(lbi)){
				nOfSlackVariables++;
			}
			if(!Double.isNaN(ub[i])){
				nOfSlackVariables++;
			}
		}
		int expectedS = nOfSlackVariables;
		
		//standard form conversion
		boolean strictlyStandardForm = true;
		LPStandardConverter lpConverter = new LPStandardConverter(strictlyStandardForm);
		lpConverter.toStandardForm(c, null, null, A, b, lb, ub);
		
		int n = lpConverter.getStandardN();
		int s = lpConverter.getStandardS();
		c = lpConverter.getStandardC().toArray();
		A = lpConverter.getStandardA().toArray();
		b = lpConverter.getStandardB().toArray();
		lb = lpConverter.getStandardLB().toArray();
		ub = (lpConverter.getStandardUB()==null)? null : ub;
		logger.debug("n : " + n);
		logger.debug("s : " + s);
		logger.debug("c : " + ArrayUtils.toString(c));
		logger.debug("A : " + ArrayUtils.toString(A));
		logger.debug("b : " + ArrayUtils.toString(b));
		logger.debug("lb : " + ArrayUtils.toString(lb));
		//logger.debug("ub : " + ArrayUtils.toString(ub));
		
		//check consistency
		assertEquals(expectedS, s);
		assertEquals(lb.length, n);
		assertTrue(ub == null);
		
		//check constraints
		RealMatrix AStandard = new Array2DRowRealMatrix(A);
		RealVector bStandard = new ArrayRealVector(b);
		double[] expectedStandardSol = lpConverter.getStandardComponents(expectedSol);
		RealVector expectedStandardSolVector = new ArrayRealVector(expectedStandardSol);
		
		for(int i=0; i<expectedStandardSolVector.getDimension(); i++){
			assertTrue(expectedStandardSolVector.getEntry(i) + 1.E-8 >= 0.);
		}
		
		RealVector Axmb = AStandard.operate(expectedStandardSolVector).subtract(bStandard);
		for(int i=0; i<Axmb.getDimension(); i++){
			assertEquals(0., Axmb.getEntry(i), expectedTol);
		}
		
		super.writeDoubleArrayToFile(new double[]{s}, "target" + File.separator	+ "standardS_"+problemId+".txt");
		super.writeDoubleArrayToFile(c, "target" + File.separator	+ "standardC_"+problemId+".txt");
		super.writeDoubleMatrixToFile(A, "target" + File.separator	+ "standardA_"+problemId+".csv");
		super.writeDoubleArrayToFile(b, "target" + File.separator	+ "standardB_"+problemId+".txt");
		super.writeDoubleArrayToFile(lb, "target" + File.separator	+ "standardLB_"+problemId+".txt");
		//ub is null super.writeDoubleArrayToFile(ub, "target" + File.separator	+ "standardUB_"+problemId+".txt");
	}
	
}
