/*
 * Copyright 2025-present Solver4J
 *
 * This work is licensed under the Creative Commons Attribution-NoDerivatives 4.0 
 * International License. To view a copy of this license, visit 
 *
 *        http://creativecommons.org/licenses/by-nd/4.0/ 
 *
 * or send a letter to Creative Commons, PO Box 1866, Mountain View, CA 94042, USA.
 */
package com.solver4j.solvers;

import java.io.File;
import java.util.List;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.solver4j.util.LPNetlibProblem;
import com.solver4j.util.MPSParser;
import com.solver4j.util.Solver4JBaseTest;
import com.solver4j.util.Utils;

/**
 * Standard form conversion test.
 * 
 * @author <a href="mailto:orion.waverly@gmail.com">Orion Waverly</a>
 */
public class LPStandardConverterNetlibTest extends Solver4JBaseTest {

	private Logger logger = LoggerFactory.getLogger(this.getClass().getName());
	
	/**
	 * Tests the standardization of all the netlib problems.
	 */
	public void xxxtestAllNetlib() throws Exception {
		logger.debug("testAllNetlib");
		Map<String, LPNetlibProblem> resultsMap = LPNetlibProblem.loadAllProblems();
		//List<NetlibLPResult> list = NetlibLPResult.getProblemsOrderedByName(resultsMap);
		//List<NetlibLPResult> list = NetlibLPResult.getProblemsOrderedByNumberOfRows(resultsMap);
		List<LPNetlibProblem> list = LPNetlibProblem.getProblemsOrderedByNumberOfColumns(resultsMap);
		for(LPNetlibProblem problem : list){
			logger.debug("problem: " + problem);
			logger.debug(problem.name);
			doTesting(problem.name);
		}
	}
	
	/**
	 * Tests the standardization of the afiro netlib problem.
	 */
	public void testAfiro() throws Exception {
		logger.debug("testAfiro");
	  
		String problemName = "afiro";
		doTesting(problemName);
	}
	
	/**
	 * Tests the standardization of the agg netlib problem.
	 */
	public void testAgg() throws Exception {
		logger.debug("testAgg");
	
		String problemName = "agg";
		doTesting(problemName);
	}

	/**
	 * Tests the standardization of the blend netlib problem.
	 */
	public void testBlend() throws Exception {
		logger.debug("testBlend");
	
		String problemName = "blend";
		doTesting(problemName);
	}

	/**
	 * Tests the standardization of the brandyPresolved netlib problem.
	 */
	public void testBrandyPresolved() throws Exception {
		logger.debug("testBrandyPresolved");
	
		String problemName = "brandyPresolved";
		doTesting(problemName);
	}

	/**
	 * Tests the standardization of the cre-aPresolved netlib problem.
	 */
	public void testCreaPresolved() throws Exception {
		logger.debug("testCreaPresolved");
	
		String problemName = "cre-aPresolved";
		doTesting(problemName);
	}

	/**
	 * Tests the standardization of the etamacro netlib problem.
	 */
	public void testEtamacro() throws Exception {
		logger.debug("testEtamacro");
	
		String problemName = "etamacro";
		doTesting(problemName);
	}

	/**
	 * Tests the standardization of the fit1d netlib problem.
	 */
	public void testFit1d() throws Exception {
		logger.debug("testFit1d");
	
		String problemName = "fit1d";
		doTesting(problemName);
	}

	/**
	 * Tests the standardization of the israel netlib problem.
	 */
	public void testIsrael() throws Exception {
		logger.debug("testIsrael");
	
		String problemName = "israel";
		doTesting(problemName);
	}

	/**
	 * Tests the standardization of the kb2 netlib problem.
	 */
	public void testKb2() throws Exception {
		logger.debug("testKb2");
	  
		String problemName = "kb2";
		doTesting(problemName);
	}
	
	/**
	 * Tests the standardization of the maros netlib problem.
	 */
	public void testMaros() throws Exception {
		logger.debug("testMaros");
	
		String problemName = "maros";
		doTesting(problemName);
	}

	/**
	 * Tests the standardization of the pilot4 netlib problem.
	 */
	public void testPilot4() throws Exception {
		logger.debug("testPilot4");
	
		String problemName = "pilot4";
		doTesting(problemName);
	}

	/**
	 * Tests the standardization of the recipe netlib problem.
	 */
	public void testRecipe() throws Exception {
		logger.debug("testRecipe");
	
		String problemName = "recipe";
		doTesting(problemName);
	}

	/**
	 * Tests the standardization of the sc50a netlib problem.
	 */
	public void testSc50a() throws Exception {
		logger.debug("testSc50a");
	  
		String problemName = "sc50a";
		doTesting(problemName);
	}
	
	/**
	 * Tests the standardization of the sc50b netlib problem.
	 */
	public void testSc50b() throws Exception {
		logger.debug("testSc50b");
	  
		String problemName = "sc50b";
		doTesting(problemName);
	}
	
	/**
	 * Tests the standardization of the scorpion netlib problem.
	 */
	public void testScorpion() throws Exception {
		logger.debug("testScorpion");
	  
		String problemName = "scorpion";
		doTesting(problemName);
	}
	
	/**
	 * Tests the standardization of the sctap1 netlib problem.
	 */
	public void testSctap1() throws Exception {
		logger.debug("testSctap1");
	  
		String problemName = "sctap1";
		doTesting(problemName);
	}
	
	/**
	 * Tests the standardization of the shell netlib problem.
	 */
	public void testShell() throws Exception {
		logger.debug("testShell");
	
		String problemName = "shell";
		doTesting(problemName);
	}

	/**
	 * Tests the standardization of the ship04s netlib problem.
	 */
	public void testShip04s() throws Exception {
		logger.debug("testShip04s");
	  
		String problemName = "ship04s";
		doTesting(problemName);
	}
	
	/**
	 * Tests the standardization of the ship08l netlib problem.
	 */
	public void testShip08l() throws Exception {
		logger.debug("testShip08l");
	
		String problemName = "ship08l";
		doTesting(problemName);
	}

	/**
	 * Tests the standardization of the sierra netlib problem.
	 */
	public void testSierra() throws Exception {
		logger.debug("testSierra");
	  
		String problemName = "sierra";
		doTesting(problemName);
	}
	
	private void doTesting(String problemName) throws Exception {
		logger.debug("doTesting: " + problemName);
		File f = Utils.getClasspathResourceAsFile("lp" + File.separator	+ "netlib" + File.separator + problemName + File.separator	+ problemName + ".mps");
		MPSParser mps = new MPSParser();
		mps.parse(f);

		//standard form conversion
		LPStandardConverter lpParser = new LPStandardConverter();
		lpParser.toStandardForm(mps.getC(), mps.getG(), mps.getH(), mps.getA(), mps.getB(), mps.getLb(), mps.getUb());
		
		int n = lpParser.getStandardN();
		int s = lpParser.getStandardS();
		double[] c = lpParser.getStandardC().toArray();
		double[][] A = lpParser.getStandardA().toArray();
		double[] b = lpParser.getStandardB().toArray();
		double[] lb = lpParser.getStandardLB().toArray();
		double[] ub = lpParser.getStandardUB().toArray();
		logger.debug("n : " + n);
		logger.debug("s : " + s);
		
		assertEquals(mps.getG().rows(), s);
		assertEquals(s + lpParser.getOriginalN(), n);
		assertEquals(lb.length, n);
		assertEquals(ub.length, n);
		
//			String outDir = "." + File.separator	+ "src" + File.separator	+ "test" + File.separator	+ "resources"+ File.separator	+"lp" + File.separator	+ "netlib";  
//			super.writeDoubleArrayToFile(new double[]{s}, outDir + File.separator + problem.name + File.separator	+ "standardS.txt");
//			super.writeDoubleArrayToFile(c, outDir + File.separator + problem.name + File.separator	+ "standardC.txt");
//			super.writeDoubleMatrixToFile(A, outDir + File.separator + problem.name + File.separator	+ "standardA.txt");
//			super.writeDoubleArrayToFile(b, outDir + File.separator + problem.name + File.separator	+ "standardB.txt");
//			super.writeDoubleArrayToFile(lb, outDir + File.separator + problem.name + File.separator	+ "standardLB.txt");
//			super.writeDoubleArrayToFile(ub, outDir + File.separator + problem.name + File.separator	+ "standardUB.txt");
			
	}
}
