/*
 * Copyright 2025-present Solver4J
 *
 * This work is licensed under the Creative Commons Attribution-NoDerivatives 4.0 
 * International License. To view a copy of this license, visit 
 *
 *        http://creativecommons.org/licenses/by-nd/4.0/ 
 *
 * or send a letter to Creative Commons, PO Box 1866, Mountain View, CA 94042, USA.
 */
package com.solver4j.solvers;

import java.io.File;

import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.math3.linear.ArrayRealVector;
import org.apache.commons.math3.linear.MatrixUtils;
import org.apache.commons.math3.linear.RealMatrix;
import org.apache.commons.math3.linear.RealVector;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.solver4j.exception.InfeasibleProblemException;
import com.solver4j.util.Solver4JBaseTest;
import com.solver4j.util.Utils;

/**
 * @author <a href="mailto:orion.waverly@gmail.com">Orion Waverly</a>
 */
public class LPPrimalDualMethodTest extends Solver4JBaseTest {

	private Logger logger = LoggerFactory.getLogger(this.getClass().getName());

	public void testDummy() throws Exception {
		logger.debug("testDummy");
	}
	
	/**
	 * Simple problem in the form
	 * min(-100x + y) s.t.
	 * x - y = 0
	 * 0 <= x <= 1
	 * 0 <= y <= 1
	 * 
	 */
	public void testSimple1() throws Exception {
		logger.debug("testSimple1");
		
		double[] c = new double[] { -100, 1 };
		double[][] A = new double[][] {{1, -1}};
		double[] b = new double[] {0};
		double[] lb = new double[] {0, 0};
		double[] ub = new double[] {1, 1};
		double minLb = LPPrimalDualMethod.DEFAULT_MIN_LOWER_BOUND;
		double maxUb = LPPrimalDualMethod.DEFAULT_MAX_UPPER_BOUND;
		
		LPOptimizationRequest or = new LPOptimizationRequest();
		or.setC(c);
		or.setA(A);
		or.setB(b);
		or.setLb(lb);
		or.setUb(ub);
		or.setCheckKKTSolutionAccuracy(true);
		or.setToleranceFeas(1.E-7);
		or.setTolerance(1.E-7);
		or.setDumpProblem(true);
		//or.setPresolvingDisabled(true);
		or.setRescalingDisabled(true);
		
		//optimization
		LPPrimalDualMethod opt = new LPPrimalDualMethod(minLb, maxUb);
		
		opt.setLPOptimizationRequest(or);
		opt.optimize();
		
		LPOptimizationResponse response = opt.getLPOptimizationResponse();
		double[] sol = response.getSolution();
		RealVector cVector = new ArrayRealVector(c);
		RealVector solVector = new ArrayRealVector(sol);
		double value = cVector.dotProduct(solVector);
		logger.debug("sol   : " + ArrayUtils.toString(sol));
		logger.debug("value : "	+ value);
		
		assertEquals( 2, sol.length);
		assertEquals( 1, sol[0], or.getTolerance());
		assertEquals( 1, sol[1], or.getTolerance());
		assertEquals(-99, value, or.getTolerance());
	}
	
	/**
	 * Simple problem in the form
	 * min(c.x) s.t.
	 * A.x = b
	 * x >=0
	 */
	public void testSimple2() throws Exception {
		logger.debug("testSimple2");
		
		double[] c = new double[] { -1, -2 };
		double[][] A = new double[][] {{1, 1}};
		double[] b = new double[] {1};
		
		LPOptimizationRequest or = new LPOptimizationRequest();
		or.setC(c);
		or.setA(A);
		or.setB(b);
		or.setLb(new double[]{0, 0});
		//or.setUb(new double[]{Double.NaN, Double.NaN});
		//or.setInitialPoint(new double[] { 0.9, 0.1 });
		//or.setNotFeasibleInitialPoint(new double[] { -0.5, 1.5 });
		or.setCheckKKTSolutionAccuracy(true);
		or.setToleranceFeas(1.E-7);
		or.setTolerance(1.E-7);
		or.setDumpProblem(true);
		//or.setPresolvingDisabled(true);
		
		//optimization
		LPPrimalDualMethod opt = new LPPrimalDualMethod();
		
		opt.setLPOptimizationRequest(or);
		opt.optimize();
		
		LPOptimizationResponse response = opt.getLPOptimizationResponse();
		double[] sol = response.getSolution();
		RealVector cVector = new ArrayRealVector(c);
		RealVector solVector = new ArrayRealVector(sol);
		double value = cVector.dotProduct(solVector);
		logger.debug("sol   : " + ArrayUtils.toString(sol));
		logger.debug("value : "	+ value);
		
		assertEquals( 2, sol.length);
		assertEquals( 0, sol[0], or.getTolerance());
		assertEquals( 1, sol[1], or.getTolerance());
		assertEquals(-2, value, or.getTolerance());
	}
	
	/**
	 * Simple problem in the form
	 * min(c.x) s.t.
	 * A.x = b
	 * lb <= x <= ub
	 * with a free variable.
	 * This test shows that it is necessary to provide bounds for all the variables in order to avoid singular KKT systems.
	 */
	public void testSimple3() throws Exception {
		logger.debug("testSimple3");
		
		double[] c = new double[] { -1, -2, 0 };
		double[][] A = new double[][] {{1, 1, 0}};
		double[] b = new double[] {1};
		//double minLb = LPPrimalDualMethod.DEFAULT_MIN_LOWER_BOUND;
		//double maxUb = LPPrimalDualMethod.DEFAULT_MAX_UPPER_BOUND;
		double minLb = -99;
		double maxUb = +99;
		
		LPOptimizationRequest or = new LPOptimizationRequest();
		or.setC(c);
		or.setA(A);
		or.setB(b);
		or.setLb(new double[]{-1, -1, -100});//this will be limited to minLb
		or.setUb(new double[]{ 1,  1,  100});//this will be limited to maxUb
		or.setCheckKKTSolutionAccuracy(true);
//		or.setToleranceFeas(1.E-7);
//		or.setTolerance(1.E-7);
		or.setDumpProblem(true);
		//or.setPresolvingDisabled(true);
		//or.setRescalingDisabled(true);
		
		//optimization
		LPPrimalDualMethod opt = new LPPrimalDualMethod(minLb, maxUb);
		opt.setLPOptimizationRequest(or);
		opt.optimize();
		
		LPOptimizationResponse response = opt.getLPOptimizationResponse();
		double[] sol = response.getSolution();
		RealVector cVector = new ArrayRealVector(c);
		RealVector solVector = new ArrayRealVector(sol);
		double value = cVector.dotProduct(solVector);
		logger.debug("sol   : " + ArrayUtils.toString(sol));
		logger.debug("value : "	+ value);
		
		assertEquals( 3, sol.length);
		assertEquals( 0, sol[0], or.getTolerance());
		assertEquals( 1, sol[1], or.getTolerance());
		assertEquals(-2, value,  or.getTolerance());
	}
	
	/**
	 * Minimize x subject to 
	 * x+y=4, 
	 * x-y=2. 
	 * Should return (3,1).
	 * This problem is the same as NewtonLEConstrainedISPTest.testOptimize2()
	 * and can be solved only with the use of a linear presolving phase:
	 * if passed directly to the solver, it will fail because Solver4J
	 * does not want rank-deficient inequalities matrices like that of this problem.
	 */
	public void testSimple4() throws Exception {
		logger.debug("testSimple4");
		double[] c = new double[] { 1, 0 };
		double[][] A = new double[][] { { 1.0, 1.0 }, { 1.0, -1.0 } };
		double[] b = new double[] { 4.0, 2.0 };
		
		LPOptimizationRequest or = new LPOptimizationRequest();
		or.setC(c);
		or.setA(A);
		or.setB(b);
		or.setLb(new double[]{-100, -100});
		or.setUb(new double[]{ 100,  100});
		or.setCheckKKTSolutionAccuracy(true);
		or.setDumpProblem(true);
		
		//optimization
		LPPrimalDualMethod opt = new LPPrimalDualMethod();
		opt.setLPOptimizationRequest(or);
		opt.optimize();
		
		LPOptimizationResponse response = opt.getLPOptimizationResponse();
		double[] sol = response.getSolution();
		RealVector cVector = new ArrayRealVector(c);
		RealVector solVector = new ArrayRealVector(sol);
		double value = cVector.dotProduct(solVector);
		logger.debug("sol   : " + ArrayUtils.toString(sol));
		logger.debug("value : "	+ value);
		
		assertEquals( 2, sol.length);
		assertEquals(3.0, sol[0], or.getTolerance());
		assertEquals(1.0, sol[1], or.getTolerance());
	}
	
	/**
	 * Simple problem in the form
	 * min(-10*x -10*y) s.t.
	 * 0 <= x <= 1
	 * 0 <= y <= 1
	 * NB: no equalities, no inequalities, just bounds	 
	 */
	public void testSimple5() throws Exception {
		logger.debug("testSimple5");
		
		double[] c = new double[] { -10, -10 };
		double[] lb = new double[] {0, 0};
		double[] ub = new double[] {1, 1};
		double minLb = LPPrimalDualMethod.DEFAULT_MIN_LOWER_BOUND;
		double maxUb = LPPrimalDualMethod.DEFAULT_MAX_UPPER_BOUND;
		
		LPOptimizationRequest or = new LPOptimizationRequest();
		or.setC(c);
		or.setLb(lb);
		or.setUb(ub);
		or.setCheckKKTSolutionAccuracy(true);
		or.setToleranceFeas(1.E-7);
		or.setTolerance(1.E-7);
		or.setDumpProblem(true);
		//or.setPresolvingDisabled(true);
		or.setRescalingDisabled(true);
		
		//optimization
		LPPrimalDualMethod opt = new LPPrimalDualMethod(minLb, maxUb);
		
		opt.setLPOptimizationRequest(or);
		opt.optimize();
		
		LPOptimizationResponse response = opt.getLPOptimizationResponse();
		double[] sol = response.getSolution();
		RealVector cVector = new ArrayRealVector(c);
		RealVector solVector = new ArrayRealVector(sol);
		double value = cVector.dotProduct(solVector);
		logger.debug("sol   : " + ArrayUtils.toString(sol));
		logger.debug("value : "	+ value);
		
		assertEquals( 2, sol.length);
		assertEquals( 1, sol[0], or.getTolerance());
		assertEquals( 1, sol[1], or.getTolerance());
		assertEquals( -20, value, or.getTolerance());
	}

	/**
	 * Simple problem in the form
	 * max( x1 - 3*x2) s.t.
	 * 2*x1 + x2 < 10;
	 * x1 - x2 > 3
	 * 0<= x1 <= 10
	 * 0<= x2 <= 10
	 * Submitted 10/01/2017 by Zaguia Rashad.	 
	 */
	public void testSimple6() throws Exception {
		logger.debug("testSimple6");
		
		double[] c = new double[] { -1, 3 };
		double[][] G = new double[][] {{2, 1}, {-1, 1}};
		double[] h = new double[] {10, -3};
		double[] lb = new double[] {0, 0};
		double[] ub = new double[] {10, 10};
		
		LPOptimizationRequest or = new LPOptimizationRequest();
		or.setC(c);
		or.setG(G);
		or.setH(h);
		or.setLb(lb);
		or.setUb(ub);
		or.setDumpProblem(true);
		
		//optimization
		LPPrimalDualMethod opt = new LPPrimalDualMethod();
		opt.setLPOptimizationRequest(or);
		opt.optimize();
		
		LPOptimizationResponse response = opt.getLPOptimizationResponse();
		double[] sol = response.getSolution();
		RealVector cVector = new ArrayRealVector(c);
		RealVector solVector = new ArrayRealVector(sol);
		double value = cVector.dotProduct(solVector);
		logger.debug("sol   : " + ArrayUtils.toString(sol));
		logger.debug("value : "	+ value);
		
		assertEquals( 2, sol.length);
		assertEquals( 5, sol[0], or.getTolerance());
		assertEquals( 0, sol[1], or.getTolerance());
		assertEquals( -5, value, or.getTolerance());
	}

	/**
	 * Linear programming in 2D in LP form.
	 * This is the same problem as {@link Solver4JTest#testLinearProgramming2D()} solved with LPPrimalDualMethod.
	 */
	public void testLPLinearProgramming2D() throws Exception {
		logger.debug("testLPLinearProgramming2D");
		
		// START SNIPPET: LPLinearProgramming-1
		
		//Objective function
		double[] c = new double[] { -1., -1. };
		
		//Inequalities constraints
		double[][] G = new double[][] {{4./3., -1}, {-1./2., 1.}, {-2., -1.}, {1./3., 1.}};
		double[] h = new double[] {2., 1./2., 2., 1./2.};
		
		//Bounds on variables
		double[] lb = new double[] {0 , 0};
		double[] ub = new double[] {10, 10};
		
		//optimization problem
		LPOptimizationRequest or = new LPOptimizationRequest();
		or.setC(c);
		or.setG(G);
		or.setH(h);
		or.setLb(lb);
		or.setUb(ub);
		or.setDumpProblem(true); 
		
		//optimization
		LPPrimalDualMethod opt = new LPPrimalDualMethod();
		
		opt.setLPOptimizationRequest(or);
		opt.optimize();
		
		// END SNIPPET: LPLinearProgramming-1
		
		OptimizationResponse response = opt.getOptimizationResponse();
		double[] sol = response.getSolution();
		RealVector cVector = new ArrayRealVector(c);
		RealVector solVector = new ArrayRealVector(sol);
		double value = cVector.dotProduct(solVector);
		logger.debug("sol   : " + ArrayUtils.toString(sol));
		logger.debug("value : "	+ value);
		assertEquals(1.5, sol[0], or.getTolerance());
		assertEquals(0.0, sol[1], or.getTolerance());
  }
	
	/**
	 * Infeasible problem in the form 
	 * min(c) s.t. 
	 * G.x < h 
	 * A.x = b 
	 * lb <= x <= ub
	 * 
	 * Submitted  26/03/2016 by Adriaan Joubert.
	 */
	public void testInfeasible1() throws Exception {
		logger.debug("testInfeasible1");

		double[] c = new double[] { 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0 };
		double[][] A = new double[][] { { 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0 } };
		double[] b = new double[] { 1.0 };
		double[][] G = new double[][] { 
				{ 1.0, 1.0, 0.0, 0.0, 0.0, 0.0, 1.0, 1.0 },
				{ 1.0, 0.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0 } };
		double[] h = new double[] { 0.3, 0.4 };
		double[] lb = new double[] { 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0 };
		double[] ub = new double[] { 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0 };
		
		LPOptimizationRequest or = new LPOptimizationRequest();
		or.setC(c);
		or.setA(A);
		or.setB(b);
		or.setG(G);
		or.setH(h);
		or.setLb(lb);
		or.setUb(ub);
		or.setDumpProblem(true);

		LPPrimalDualMethod opt = new LPPrimalDualMethod();
		opt.setLPOptimizationRequest(or);

		try{
			opt.optimize();
			//unexpected behavior: the problem is infeasible
			fail("the problem is expected to be infeasible");
			
		}catch(InfeasibleProblemException e){
			//expected behavior: the problem is infeasible 
			assertTrue(true);
		}
	}
	
	/**
	 * Infeasible problem. 
	 * This is a reduced formulation of {@link LPPrimalDualMethodTest#testInfeasible1()}.
	 */
	public void testInfeasible1Red() throws Exception {
		logger.debug("testInfeasible1Red");

		double[] c = new double[] { 1.0, 1.0, 1.0 };
		double[][] A = new double[][] { { 1.0, 1.0, 1.0 } };
		double[] b = new double[] { 1.0 };
		double[][] G = new double[][] { 
				{ 1.0, 1.0, 0.0 },
				{ 0.0, 1.0, 1.0 } };
		double[] h = new double[] { 0.3, 0.4 };
		double[] lb = new double[] { 0.0, 0.0, 0.0 };
		double[] ub = new double[] { 1.0, 1.0, 1.0 };

		LPOptimizationRequest or = new LPOptimizationRequest();
		or.setC(c);
		or.setA(A);
		or.setB(b);
		or.setG(G);
		or.setH(h);
		or.setLb(lb);
		or.setUb(ub);
		or.setDumpProblem(true);

		LPPrimalDualMethod opt = new LPPrimalDualMethod();
		opt.setLPOptimizationRequest(or);

		try{
			opt.optimize();
			
			//unexpected behavior: the problem is infeasible
			fail("the problem is expected to be infeasible");
			
		}catch(InfeasibleProblemException e){
			//expected behavior: the problem is infeasible 
		}
	}
	
	/**
	 * Problem in the form
	 * min(c.x) s.t.
	 * G.x < h
	 * A.x = b
	 * 
	 * This is a good for testing with a small size problem.
	 * Submitted 01/09/2013 by Chris Myers.
	 * 
	 * @see Solver4JTest#testCGhAb1()
	 */
	public void testCGhAb1() throws Exception {
		logger.debug("testCGhAb1");
		
		String problemId = "1";
		
		//the original problem: ok until precision 1.E-7
		double[] c = super.loadDoubleArrayFromFile("lp"+File.separator+"c"+problemId+".txt");
		double[][] G = super.loadDoubleMatrixFromFile("lp"+File.separator+"G"+problemId+".csv", ",".charAt(0));
		double[] h = super.loadDoubleArrayFromFile("lp"+File.separator+"h"+problemId+".txt");;
		double[][] A = super.loadDoubleMatrixFromFile("lp"+File.separator+"A"+problemId+".csv", ",".charAt(0));
		double[] b = super.loadDoubleArrayFromFile("lp"+File.separator+"b"+problemId+".txt");
		double[] expectedSol = super.loadDoubleArrayFromFile("lp"+File.separator+"sol"+problemId+".txt");
		double expectedvalue = super.loadDoubleArrayFromFile("lp"+File.separator+"value"+problemId+".txt")[0];
		
		//double norm = MatrixUtils.createRealMatrix(A).operate(MatrixUtils.createRealVector(expectedSol)).subtract(MatrixUtils.createRealVector(b)).getNorm();
		//assertTrue(norm < 1.e-10);
		
		LPOptimizationRequest or = new LPOptimizationRequest();
		or.setC(c);
		or.setG(G);
		or.setH(h);
		or.setA(A);
		or.setB(b);
		or.setCheckKKTSolutionAccuracy(true);
		or.setToleranceKKT(1.E-7);
		or.setToleranceFeas(1.E-7);
		or.setTolerance(1.E-7);
		or.setDumpProblem(true);
		or.setAlpha(0.75);
		or.setInitialPoint(new double[]{0.9999998735888544,-999.0000001264111,1000.0,0.9999998735888544,0.0,-999.0000001264111,0.9999999661257591,0.9999998735888544,1000.0,0.0,0.9999998735888544,0.0,0.9999998735888544,0.9999998735888544,0.9999998735888544,0.0,0.0,0.9999998735888544,-1000.0,0.9999999198573067,9.253690467190285E-8,1000.0,-999.0000001264111,0.9999998735888544,-1000.0,-1000.0});
		
		//optimization
		//LPPrimalDualMethodOLD opt = new LPPrimalDualMethodOLD();
		LPPrimalDualMethod opt = new LPPrimalDualMethod();
		
		opt.setLPOptimizationRequest(or);
		opt.optimize();
		
		LPOptimizationResponse response = opt.getLPOptimizationResponse();
		double[] sol = response.getSolution();
		RealVector cVector = new ArrayRealVector(c);
		RealVector solVector = new ArrayRealVector(sol);
		double value = cVector.dotProduct(solVector);
		logger.debug("sol   : " + ArrayUtils.toString(sol));
		logger.debug("value : "	+ value);
		
		//check constraints
		RealVector x = MatrixUtils.createRealVector(sol);
		RealMatrix GMatrix = MatrixUtils.createRealMatrix(G); 
		RealVector hvector = MatrixUtils.createRealVector(h);
		RealMatrix AMatrix = MatrixUtils.createRealMatrix(A); 
		RealVector bvector = MatrixUtils.createRealVector(b);
		RealVector Gxh = GMatrix.operate(x).subtract(hvector);
		for(int i=0; i<Gxh.getDimension(); i++){
			assertTrue(Gxh.getEntry(i)<=0);//not strictly because some constraint has been treated as a bound
		}
		RealVector Axb = AMatrix.operate(x).subtract(bvector);
		assertEquals(0., Axb.getNorm(), or.getToleranceFeas());
		
		//check value
		assertEquals(expectedvalue, value, or.getTolerance());
		
	}
	
	/**
	 * Problem in the form
	 * min(c.x) s.t.
	 * G.x < h
	 * A.x = b
	 * lb <= x <= ub
	 * 
	 * This is the same as testCGhAb3, but lb and ub are outside G.
	 * The presolved problem has a deterministic solution, that is, all the variables have a fixed value.
	 * Submitted 01/09/2013 by Chris Myers.
	 */
	public void testCGhAbLbUb2() throws Exception {
		logger.debug("testCGhAbLbUb2");
		
		String problemId = "2";
		
		logger.debug("problemId: " + problemId);
		double[] c = super.loadDoubleArrayFromFile("lp"+File.separator+"c"+problemId+".txt");
		double[][] G = super.loadDoubleMatrixFromFile("lp"+File.separator+"G"+problemId+".csv", ",".charAt(0));
		double[] h = super.loadDoubleArrayFromFile("lp"+File.separator+"h"+problemId+".txt");;
		double[][] A = super.loadDoubleMatrixFromFile("lp"+File.separator+"A"+problemId+".csv", ",".charAt(0));
		double[] b = super.loadDoubleArrayFromFile("lp"+File.separator+"b"+problemId+".txt");
		double[] lb = super.loadDoubleArrayFromFile("lp"+File.separator+"lb"+problemId+".txt");
		double[] ub = super.loadDoubleArrayFromFile("lp"+File.separator+"ub"+problemId+".txt");
		double[] expectedSol = super.loadDoubleArrayFromFile("lp"+File.separator+"sol"+problemId+".txt");
		double expectedvalue = super.loadDoubleArrayFromFile("lp"+File.separator+"value"+problemId+".txt")[0];
		double minLb = 0;
		double maxUb = 1.0E15;//it is do high because of the very high values of the elements of h
		
		LPOptimizationRequest or = new LPOptimizationRequest();
		or.setC(c);
		or.setG(G);
		or.setH(h);
		or.setA(A);
		or.setB(b);
		or.setLb(lb);
		or.setUb(ub);
		//or.setInitialPoint(new double[] {100000.00000000377, 2000000.0000000752, 100000.00000000095, 2000000.0000000189, 100000.00000000095, 2000000.0000000189, 100000.00000000095, 2000000.0000000189, 100000.00000000095, 2000000.0000000189});
		or.setCheckKKTSolutionAccuracy(true);
		//or.setToleranceKKT(1.e-5);
		//or.setToleranceFeas(5.E-5);
		//or.setTolerance(1.E-7);
		or.setDumpProblem(true);
		//or.setPresolvingDisabled(true);
		
		//optimization
		LPPrimalDualMethod opt = new LPPrimalDualMethod(minLb, maxUb);
		
		opt.setLPOptimizationRequest(or);
		opt.optimize();
		
		LPOptimizationResponse response = opt.getLPOptimizationResponse();
		double[] sol = response.getSolution();
		RealVector cVector = new ArrayRealVector(c);
		RealVector solVector = new ArrayRealVector(sol);
		double value = cVector.dotProduct(solVector);
		logger.debug("sol   : " + ArrayUtils.toString(sol));
		logger.debug("value : "	+ value);
		
		//check constraints
		assertEquals(lb.length, sol.length);
		assertEquals(ub.length, sol.length);
		RealVector x = MatrixUtils.createRealVector(sol);
		RealMatrix GMatrix = MatrixUtils.createRealMatrix(G); 
		RealVector hvector = MatrixUtils.createRealVector(h);
		RealMatrix AMatrix = MatrixUtils.createRealMatrix(A); 
		RealVector bvector = MatrixUtils.createRealVector(b);
		for(int i=0; i<lb.length; i++){
			double di = Double.isNaN(lb[i])? -Double.MAX_VALUE : lb[i];
			assertTrue(di <= x.getEntry(i));
		}
		for(int i=0; i<ub.length; i++){
			double di = Double.isNaN(ub[i])? Double.MAX_VALUE : ub[i];
			assertTrue(di>= x.getEntry(i));
		}
		RealVector Gxh = GMatrix.operate(x).subtract(hvector);
		for(int i=0; i<Gxh.getDimension(); i++){
			assertTrue(Gxh.getEntry(i)<0);
		}
		RealVector Axb = AMatrix.operate(x).subtract(bvector);
		assertEquals(0., Axb.getNorm(), or.getToleranceFeas());
		
//		assertEquals( expectedSol.length, sol.length);
//		for(int i=0; i<sol.length; i++){
//			assertEquals(expectedSol[0], sol[0], 1.e-7);
//		}
		
		logger.debug("expectedvalue : "	+ expectedvalue);
		//assertEquals(expectedvalue, value, or.getTolerance());
		assertTrue(expectedvalue > value);
	
	}
	
	/**
	 * Problem in the form
	 * min(c.x) s.t.
	 * G.x < h
	 * A.x = b
	 * 
	 * This is the same as testCGhAbLbUb2, but lb and ub are into G.
	 * The presolved problem has a deterministic solution, that is, all the variables have a fixed value.
	 */
	public void testCGhAb3() throws Exception {
		logger.debug("testCGhAb3");
		
		String problemId = "3";
		
		logger.debug("problemId: " + problemId);
		double[] c = super.loadDoubleArrayFromFile("lp"+File.separator+"c"+problemId+".txt");
		double[][] G = super.loadDoubleMatrixFromFile("lp"+File.separator+"G"+problemId+".csv", ",".charAt(0));
		double[] h = super.loadDoubleArrayFromFile("lp"+File.separator+"h"+problemId+".txt");;
		double[][] A = super.loadDoubleMatrixFromFile("lp"+File.separator+"A"+problemId+".csv", ",".charAt(0));
		double[] b = super.loadDoubleArrayFromFile("lp"+File.separator+"b"+problemId+".txt");
		double[] expectedSol = super.loadDoubleArrayFromFile("lp"+File.separator+"sol"+problemId+".txt");
		double expectedvalue = super.loadDoubleArrayFromFile("lp"+File.separator+"value"+problemId+".txt")[0];
		double minLb = 0;
		double maxUb = 1.0E15;//it is so high because of the very high values of the elements of h
		
		LPOptimizationRequest or = new LPOptimizationRequest();
		or.setC(c);
		or.setG(G);
		or.setH(h);
		or.setA(A);
		or.setB(b);
		or.setCheckKKTSolutionAccuracy(true);
		//or.setToleranceKKT(1.e-5);
//		or.setToleranceFeas(5.E-5);
//		or.setTolerance(1.E-7);
		or.setDumpProblem(true);
		//or.setPresolvingDisabled(true);
		
		//optimization
		LPPrimalDualMethod opt = new LPPrimalDualMethod(minLb, maxUb);
		
		opt.setLPOptimizationRequest(or);
		opt.optimize();
		
		LPOptimizationResponse response = opt.getLPOptimizationResponse();
		double[] sol = response.getSolution();
		RealVector cVector = new ArrayRealVector(c);
		RealVector solVector = new ArrayRealVector(sol);
		double value = cVector.dotProduct(solVector);
		logger.debug("sol   : " + ArrayUtils.toString(sol));
		logger.debug("value : "	+ value);
		
		//check constraints
		RealVector x = MatrixUtils.createRealVector(sol);
		RealMatrix GMatrix = MatrixUtils.createRealMatrix(G); 
		RealVector hvector = MatrixUtils.createRealVector(h);
		RealMatrix AMatrix = MatrixUtils.createRealMatrix(A); 
		RealVector bvector = MatrixUtils.createRealVector(b);
		RealVector Gxh = GMatrix.operate(x).subtract(hvector);
		for(int i=0; i<Gxh.getDimension(); i++){
			assertTrue(Gxh.getEntry(i)<=0);
		}
		RealVector Axb = AMatrix.operate(x).subtract(bvector);
		assertEquals(0., Axb.getNorm(), or.getToleranceFeas());
		
		assertEquals( expectedSol.length, sol.length);
//		for(int i=0; i<sol.length; i++){
//			assertEquals(expectedSol[0], sol[0], or.getTolerance());
//		}
		assertEquals(expectedvalue, value, or.getTolerance());
	
	}
	
	/**
	 * Problem in the form
	 * min(c.x) s.t.
	 * G.x < h
	 * The objective function of the presolved problem has a 0-gradient.
	 */
	public void testCGh4() throws Exception {
		logger.debug("testCGh4");
		
		String problemId = "4";
		
		//the original problem: ok until precision 1.E-7
		double[] c = super.loadDoubleArrayFromFile("lp"+File.separator+"c"+problemId+".txt");
		double[][] G = super.loadDoubleMatrixFromFile("lp"+File.separator+"G"+problemId+".csv", ",".charAt(0));
		double[] h = super.loadDoubleArrayFromFile("lp"+File.separator+"h"+problemId+".txt");;
		double[] expectedSol = super.loadDoubleArrayFromFile("lp"+File.separator+"sol"+problemId+".txt");
		double expectedValue = super.loadDoubleArrayFromFile("lp"+File.separator+"value"+problemId+".txt")[0];
		
		//double norm = MatrixUtils.createRealMatrix(A).operate(MatrixUtils.createRealVector(expectedSol)).subtract(MatrixUtils.createRealVector(b)).getNorm();
		//assertTrue(norm < 1.e-10);
		
		LPOptimizationRequest or = new LPOptimizationRequest();
		or.setC(c);
		or.setG(G);
		or.setH(h);
		or.setCheckKKTSolutionAccuracy(true);
		or.setToleranceKKT(1.E-7);
		or.setToleranceFeas(1.E-6);
		or.setTolerance(1.E-5);
		or.setDumpProblem(true);
		or.setRescalingDisabled(true);//this fails with false
		
		//optimization
		//LPPrimalDualMethodOLD opt = new LPPrimalDualMethodOLD();
		LPPrimalDualMethod opt = new LPPrimalDualMethod();
		
		opt.setLPOptimizationRequest(or);
		opt.optimize();
		
		LPOptimizationResponse response = opt.getLPOptimizationResponse();
		double[] sol = response.getSolution();
		RealVector cVector = new ArrayRealVector(c);
		RealVector solVector = new ArrayRealVector(sol);
		double value = cVector.dotProduct(solVector);
		logger.debug("sol   : " + ArrayUtils.toString(sol));
		logger.debug("value : "	+ value);
		
		//check constraints
		RealVector x = MatrixUtils.createRealVector(sol);
		RealMatrix GMatrix = MatrixUtils.createRealMatrix(G); 
		RealVector hvector = MatrixUtils.createRealVector(h);
		RealVector Gxh = GMatrix.operate(x).subtract(hvector);
		for(int i=0; i<Gxh.getDimension(); i++){
			assertTrue(Gxh.getEntry(i)<=0);//not strictly because some constraint has been treated as a bound
		}
		//check value
		assertEquals(expectedValue, value, or.getTolerance());
		
	}
	
	/**
	 * Linear problem in the form 
	 * min(c.x) s.t. 
	 * G.x < h 
	 * lb < x < ub
	 * 
	 * Submitted by Philip Walton 12/11/2014
	 */
	public void testCGhLbUb9() throws Exception {
		logger.debug("testCGhLbUb9");
		
		String problemId = "9";
		
		//the original problem: ok until precision 1.E-7
		double[] c = super.loadDoubleArrayFromFile("lp"+File.separator+"c"+problemId+".txt");
		double[][] G = super.loadDoubleMatrixFromFile("lp"+File.separator+"G"+problemId+".csv", ",".charAt(0));
		double[] h = super.loadDoubleArrayFromFile("lp"+File.separator+"h"+problemId+".txt");;
		double[] lb = super.loadDoubleArrayFromFile("lp"+File.separator+"lb"+problemId+".txt");;
		double[] ub = super.loadDoubleArrayFromFile("lp"+File.separator+"ub"+problemId+".txt");;
		double[] expectedSol = super.loadDoubleArrayFromFile("lp"+File.separator+"sol"+problemId+".txt");
		double expectedValue = super.loadDoubleArrayFromFile("lp"+File.separator+"value"+problemId+".txt")[0];

		LPOptimizationRequest or = new LPOptimizationRequest();
		or.setC(c);
		or.setG(G);
		or.setH(h);
		or.setLb(lb);
		or.setUb(ub);
		//or.setToleranceFeas(1.E-9);
		//or.setTolerance(1.E-9);
		//or.setCheckKKTSolutionAccuracy(true);
		or.setDumpProblem(true);
		//or.setObjectiveFunctionDisabled(true);
		
		//optimization
		double minLb = -150000;
		double maxUb = +150000;
		LPPrimalDualMethod opt = new LPPrimalDualMethod(minLb, maxUb);
		opt.setLPOptimizationRequest(or);
		opt.optimize();
		
		LPOptimizationResponse response = opt.getLPOptimizationResponse();
		double[] sol = response.getSolution();
		RealVector cVector = new ArrayRealVector(c);
		RealVector solVector = new ArrayRealVector(sol);
		double value = cVector.dotProduct(solVector);
		logger.debug("sol   : " + ArrayUtils.toString(sol));
		logger.debug("value : "	+ value);
		
		//assertEquals(expectedvalue, value, or.getTolerance());
		assertTrue(Math.abs(expectedValue - value) / expectedValue < or.getTolerance());
	}
	
	/**
	 * Problem in the form
	 * min(c.x) s.t.
	 * G.x < h
	 * lb <= x <= ub
	 */
	public void testCGhLbUb13() throws Exception {
		logger.debug("testCGhLbUb13");
		
		String problemId = "13";
		
		double[] c = super.loadDoubleArrayFromFile("lp"+File.separator+"c"+problemId+".txt");
		double[][] G = super.loadDoubleMatrixFromFile("lp"+File.separator+"G"+problemId+".csv", ",".charAt(0));
		double[] h = super.loadDoubleArrayFromFile("lp"+File.separator+"h"+problemId+".txt");;
		double[] lb = super.loadDoubleArrayFromFile("lp"+File.separator+"lb"+problemId+".txt");
		double[] ub = super.loadDoubleArrayFromFile("lp"+File.separator+"ub"+problemId+".txt");
		double[] expectedSol = super.loadDoubleArrayFromFile("lp"+File.separator+"sol"+problemId+".txt");
		double expectedValue = super.loadDoubleArrayFromFile("lp"+File.separator+"value"+problemId+".txt")[0];
		
		LPOptimizationRequest or = new LPOptimizationRequest();
		or.setC(c);
		or.setG(G);
		or.setH(h);
		or.setLb(lb);
		or.setUb(ub);
		or.setCheckKKTSolutionAccuracy(true);
		or.setDumpProblem(true);
		
		//optimization
		LPPrimalDualMethod opt = new LPPrimalDualMethod();
		opt.setLPOptimizationRequest(or);
		opt.optimize();
		
		LPOptimizationResponse response = opt.getLPOptimizationResponse();
		double[] sol = response.getSolution();
		RealVector cVector = new ArrayRealVector(c);
		RealVector solVector = new ArrayRealVector(sol);
		double value = cVector.dotProduct(solVector);
		logger.debug("sol   : " + ArrayUtils.toString(sol));
		logger.debug("value : "	+ value);
		
		//check constraints
		RealVector x = MatrixUtils.createRealVector(sol);
		RealMatrix GMatrix = MatrixUtils.createRealMatrix(G); 
		RealVector hvector = MatrixUtils.createRealVector(h);
		RealVector Gxh = GMatrix.operate(x).subtract(hvector);
		for(int i=0; i<Gxh.getDimension(); i++){
			assertTrue(Gxh.getEntry(i) < 0);
		}
		//check value
		assertTrue((expectedValue - value) / expectedValue < or.getTolerance());
		
	}
	
	/**
	 * Problem in the form
	 * min(c.x) s.t.
	 * A.x = b
	 * lb <= x <= ub
	 * 
	 * This problem involves recursive column duplicate reductions.
	 * This is a good for testing with a small size problem.
	 */
	public void testCAbLbUb5() throws Exception {
		logger.debug("testCAbLbUb5");
		
		String problemId = "5";
		
		logger.debug("problemId: " + problemId);
		double[] c = super.loadDoubleArrayFromFile("lp"+File.separator+"c"+problemId+".txt");
		double[][] A = super.loadDoubleMatrixFromFile("lp"+File.separator+"A"+problemId+".csv", ",".charAt(0));
		double[] b = super.loadDoubleArrayFromFile("lp"+File.separator+"b"+problemId+".txt");
		double[] lb = super.loadDoubleArrayFromFile("lp"+File.separator+"lb"+problemId+".txt");
		double[] ub = super.loadDoubleArrayFromFile("lp"+File.separator+"ub"+problemId+".txt");
		double[] expectedSol = super.loadDoubleArrayFromFile("lp"+File.separator+"sol"+problemId+".txt");
		double expectedValue = super.loadDoubleArrayFromFile("lp"+File.separator+"value"+problemId+".txt")[0];
		
		LPOptimizationRequest or = new LPOptimizationRequest();
		or.setC(c);
		or.setA(A);
		or.setB(b);
		or.setLb(lb);
		or.setUb(ub);
		or.setCheckKKTSolutionAccuracy(true);
//		or.setToleranceKKT(1.e-7);
//		or.setToleranceFeas(1.E-7);
//		or.setTolerance(1.E-7);
		or.setDumpProblem(true);
		
		//optimization
		LPPrimalDualMethod opt = new LPPrimalDualMethod();
		opt.setLPOptimizationRequest(or);
		opt.optimize();
		
		LPOptimizationResponse response = opt.getLPOptimizationResponse();
		double[] sol = response.getSolution();
		RealVector cVector = new ArrayRealVector(c);
		RealVector solVector = new ArrayRealVector(sol);
		double value = cVector.dotProduct(solVector);
		logger.debug("sol   : " + ArrayUtils.toString(sol));
		logger.debug("value : "	+ value);
		
		//check constraints
		assertEquals(lb.length, sol.length);
		assertEquals(ub.length, sol.length);
		RealVector x = MatrixUtils.createRealVector(sol);
		RealMatrix AMatrix = MatrixUtils.createRealMatrix(A); 
		RealVector bvector = MatrixUtils.createRealVector(b);
		for(int i=0; i<lb.length; i++){
			double di = Double.isNaN(lb[i])? -Double.MAX_VALUE : lb[i];
			assertTrue(di <= x.getEntry(i));
		}
		for(int i=0; i<ub.length; i++){
			double di = Double.isNaN(ub[i])? Double.MAX_VALUE : ub[i];
			assertTrue(di >= x.getEntry(i));
		}
		RealVector Axb = AMatrix.operate(x).subtract(bvector);
		assertEquals(0., Axb.getNorm(), or.getToleranceFeas());
		
		//check value
		assertEquals(expectedValue, value, or.getTolerance());
	}
	
	/**
	 * Problem in the form
	 * min(c.x) s.t.
	 * A.x = b
	 * lb <= x <= ub
	 * 
	 * This problem involves column duplicate reduction.
	 * This is a good for testing with a small size problem.
	 */
	public void testCAbLbUb6() throws Exception {
		logger.debug("testCAbLbUb6");
		
		String problemId = "6";
		
		logger.debug("problemId: " + problemId);
		double[] c = super.loadDoubleArrayFromFile("lp"+File.separator+"c"+problemId+".txt");
		double[][] A = super.loadDoubleMatrixFromFile("lp"+File.separator+"A"+problemId+".csv", ",".charAt(0));
		double[] b = super.loadDoubleArrayFromFile("lp"+File.separator+"b"+problemId+".txt");
		double[] lb = super.loadDoubleArrayFromFile("lp"+File.separator+"lb"+problemId+".txt");
		double[] ub = super.loadDoubleArrayFromFile("lp"+File.separator+"ub"+problemId+".txt");
		double[] expectedSol = super.loadDoubleArrayFromFile("lp"+File.separator+"sol"+problemId+".txt");
		double expectedvalue = super.loadDoubleArrayFromFile("lp"+File.separator+"value"+problemId+".txt")[0];
		
		LPOptimizationRequest or = new LPOptimizationRequest();
		or.setC(c);
		or.setA(A);
		or.setB(b);
		or.setLb(lb);
		or.setUb(ub);
		or.setCheckKKTSolutionAccuracy(true);
//		or.setToleranceKKT(1.e-7);
//		or.setToleranceFeas(1.E-7);
//		or.setTolerance(1.E-7);
		or.setDumpProblem(true);
		
		//optimization
		LPPrimalDualMethod opt = new LPPrimalDualMethod();
		opt.setLPOptimizationRequest(or);
		opt.optimize();
		
		LPOptimizationResponse response = opt.getLPOptimizationResponse();
		double[] sol = response.getSolution();
		RealVector cVector = new ArrayRealVector(c);
		RealVector solVector = new ArrayRealVector(sol);
		double value = cVector.dotProduct(solVector);
		logger.debug("sol   : " + ArrayUtils.toString(sol));
		logger.debug("value : "	+ value);
		
		//check constraints
		assertEquals(lb.length, sol.length);
		assertEquals(ub.length, sol.length);
		RealVector x = MatrixUtils.createRealVector(sol);
		RealMatrix AMatrix = MatrixUtils.createRealMatrix(A); 
		RealVector bvector = MatrixUtils.createRealVector(b);
		for(int i=0; i<lb.length; i++){
			double di = Double.isNaN(lb[i])? -Double.MAX_VALUE : lb[i];
			assertTrue(di <= x.getEntry(i));
		}
		for(int i=0; i<ub.length; i++){
			double di = Double.isNaN(ub[i])? Double.MAX_VALUE : ub[i];
			assertTrue(di>= x.getEntry(i));
		}
		RealVector Axb = AMatrix.operate(x).subtract(bvector);
		assertEquals(0., Axb.getNorm(), or.getToleranceFeas());
		
		//check value
		assertEquals(expectedvalue, value, or.getTolerance());
	}
	
	/**
	 * Problem in the form
	 * min(c.x) s.t.
	 * G.x < h
	 * A.x = b
	 * lb <= x <= ub
	 * 
	 */
	public void testCGhAbLbUb7() throws Exception {
		logger.debug("testCGhAbLbUb7");
		
		String problemId = "7";
		
		logger.debug("problemId: " + problemId);
		double[] c = super.loadDoubleArrayFromFile("lp"+File.separator+"c"+problemId+".txt");
		double[][] G = super.loadDoubleMatrixFromFile("lp"+File.separator+"G"+problemId+".csv", ",".charAt(0));
		double[] h = super.loadDoubleArrayFromFile("lp"+File.separator+"h"+problemId+".txt");;
		double[][] A = super.loadDoubleMatrixFromFile("lp"+File.separator+"A"+problemId+".csv", ",".charAt(0));
		double[] b = super.loadDoubleArrayFromFile("lp"+File.separator+"b"+problemId+".txt");
		double[] lb = super.loadDoubleArrayFromFile("lp"+File.separator+"lb"+problemId+".txt");
		double[] ub = super.loadDoubleArrayFromFile("lp"+File.separator+"ub"+problemId+".txt");
		double[] expectedSol = super.loadDoubleArrayFromFile("lp"+File.separator+"sol"+problemId+".txt");
		double expectedValue = super.loadDoubleArrayFromFile("lp"+File.separator+"value"+problemId+".txt")[0];
		
		//the unbounded bounds are saved on the files with NaN values, so substitute them with acceptable values
		lb = Utils.replaceValues(lb, Double.NaN, LPPrimalDualMethod.DEFAULT_MIN_LOWER_BOUND);
		ub = Utils.replaceValues(ub, Double.NaN, LPPrimalDualMethod.DEFAULT_MAX_UPPER_BOUND);
		
		LPOptimizationRequest or = new LPOptimizationRequest();
		or.setC(c);
		or.setG(G);
		or.setH(h);
		or.setA(A);
		or.setB(b);
		or.setLb(lb);
		or.setUb(ub);
		or.setCheckKKTSolutionAccuracy(true);
//		or.setToleranceKKT(1.e-7);
//		or.setToleranceFeas(1.E-7);
//		or.setTolerance(1.E-7);
		or.setDumpProblem(true);
		//or.setPresolvingDisabled(true);
		//or.setRescalingDisabled(true);
		//or.setObjectiveFunctionDisabled(true);
		
		//optimization
		LPPrimalDualMethod opt = new LPPrimalDualMethod();
		opt.setLPOptimizationRequest(or);
		opt.optimize();
		
		LPOptimizationResponse response = opt.getLPOptimizationResponse();
		double[] sol = response.getSolution();
		RealVector cVector = new ArrayRealVector(c);
		RealVector solVector = new ArrayRealVector(sol);
		double value = cVector.dotProduct(solVector);
		logger.debug("sol   : " + ArrayUtils.toString(sol));
		logger.debug("value : "	+ value);
		
		//check constraints
		assertEquals(lb.length, sol.length);
		assertEquals(ub.length, sol.length);
		RealVector x = MatrixUtils.createRealVector(sol);
		RealMatrix GMatrix = MatrixUtils.createRealMatrix(G); 
		RealVector hvector = MatrixUtils.createRealVector(h);
		RealMatrix AMatrix = MatrixUtils.createRealMatrix(A); 
		RealVector bvector = MatrixUtils.createRealVector(b);
		for(int i=0; i<lb.length; i++){
			assertTrue(lb[i] <= x.getEntry(i));
		}
		for(int i=0; i<ub.length; i++){
			double di = Double.isNaN(lb[i])? -Double.MAX_VALUE : lb[i];
			assertTrue(di <= x.getEntry(i));
		}
		RealVector Gxh = GMatrix.operate(x).subtract(hvector);
		for(int i=0; i<Gxh.getDimension(); i++){
			double di = Double.isNaN(ub[i])? Double.MAX_VALUE : ub[i];
			assertTrue(di>= x.getEntry(i));
		}
		RealVector Axb = AMatrix.operate(x).subtract(bvector);
		assertEquals(0., Axb.getNorm(), or.getToleranceFeas());
		
		assertEquals( expectedSol.length, sol.length);
		for(int i=0; i<sol.length; i++){
			//assertEquals(expectedSol[0], sol[0], or.getTolerance());
		}
		//assertEquals(expectedvalue, value, or.getTolerance());
		assertTrue(Math.abs(expectedValue - value) / expectedValue < or.getTolerance());
	
	}
	
	/**
	 * Problem in the form
	 * min(c.x) s.t.
	 * A.x = b
	 * lb <= x <= ub
	 */
	public void testCAbLbUb8() throws Exception {
		logger.debug("testCAbLbUb8");
		
		String problemId = "8";
		
		logger.debug("problemId: " + problemId);
		double[] c = super.loadDoubleArrayFromFile("lp"+File.separator+"c"+problemId+".txt");
		double[][] A = super.loadDoubleMatrixFromFile("lp"+File.separator+"A"+problemId+".csv", ",".charAt(0));
		double[] b = super.loadDoubleArrayFromFile("lp"+File.separator+"b"+problemId+".txt");
		double[] lb = super.loadDoubleArrayFromFile("lp"+File.separator+"lb"+problemId+".txt");
		double[] ub = super.loadDoubleArrayFromFile("lp"+File.separator+"ub"+problemId+".txt");
		double[] expectedSol = super.loadDoubleArrayFromFile("lp"+File.separator+"sol"+problemId+".txt");
		double expectedValue = super.loadDoubleArrayFromFile("lp"+File.separator+"value"+problemId+".txt")[0];
		
		//the unbounded bounds are saved on the files with NaN values, so substitute them with acceptable values
		lb = Utils.replaceValues(lb, Double.NaN, LPPrimalDualMethod.DEFAULT_MIN_LOWER_BOUND);
		ub = Utils.replaceValues(ub, Double.NaN, LPPrimalDualMethod.DEFAULT_MAX_UPPER_BOUND);
		
		LPOptimizationRequest or = new LPOptimizationRequest();
		or.setC(c);
		or.setA(A);
		or.setB(b);
		or.setLb(lb);
		or.setUb(ub);
		or.setCheckKKTSolutionAccuracy(true);
		//or.setToleranceKKT(1.e-7);
		//or.setToleranceFeas(1.E-7);
		//or.setTolerance(1.E-7);
		or.setDumpProblem(true);
		//or.setObjectiveFunctionDisabled(true);
		//or.setRescalingDisabled(true);
		
		//optimization
		LPPrimalDualMethod opt = new LPPrimalDualMethod();
		opt.setLPOptimizationRequest(or);
		opt.optimize();
		
		LPOptimizationResponse response = opt.getLPOptimizationResponse();
		double[] sol = response.getSolution();
		RealVector cVector = new ArrayRealVector(c);
		RealVector solVector = new ArrayRealVector(sol);
		double value = cVector.dotProduct(solVector);
		logger.debug("sol   : " + ArrayUtils.toString(sol));
		logger.debug("value : "	+ value);
		
		//check constraints
		assertEquals(lb.length, sol.length);
		assertEquals(ub.length, sol.length);
		RealVector x = MatrixUtils.createRealVector(sol);
		RealMatrix AMatrix = MatrixUtils.createRealMatrix(A); 
		RealVector bvector = MatrixUtils.createRealVector(b);
		for(int i=0; i<lb.length; i++){
			double di = Double.isNaN(lb[i])? -Double.MAX_VALUE : lb[i];
			assertTrue(di <= x.getEntry(i));
		}
		for(int i=0; i<ub.length; i++){
			double di = Double.isNaN(ub[i])? Double.MAX_VALUE : ub[i];
			assertTrue(di>= x.getEntry(i));
		}
		RealVector Axb = AMatrix.operate(x).subtract(bvector);
		assertEquals(0., Axb.getNorm(), or.getToleranceFeas());
		
		//check value
		//assertEquals(expectedvalue, value, or.getTolerance());
		assertTrue(Math.abs(expectedValue - value) / expectedValue < or.getTolerance());
	}
	
	/**
	 * Problem in the form
	 * min(c.x) s.t.
	 * G.x < h
	 * A.x = b
	 * lb <= x <= ub
	 * 
	 */
	public void testCGhAbLbUb10() throws Exception {
		logger.debug("testCGhAbLbUb10");
		
		String problemId = "10";
		
		logger.debug("problemId: " + problemId);
		double[] c = super.loadDoubleArrayFromFile("lp"+File.separator+"c"+problemId+".txt");
		double[][] G = super.loadDoubleMatrixFromFile("lp"+File.separator+"G"+problemId+".csv", ",".charAt(0));
		double[] h = super.loadDoubleArrayFromFile("lp"+File.separator+"h"+problemId+".txt");;
		double[][] A = super.loadDoubleMatrixFromFile("lp"+File.separator+"A"+problemId+".csv", ",".charAt(0));
		double[] b = super.loadDoubleArrayFromFile("lp"+File.separator+"b"+problemId+".txt");
		double[] lb = super.loadDoubleArrayFromFile("lp"+File.separator+"lb"+problemId+".txt");
		double[] ub = super.loadDoubleArrayFromFile("lp"+File.separator+"ub"+problemId+".txt");
		double[] expectedSol = super.loadDoubleArrayFromFile("lp"+File.separator+"sol"+problemId+".txt");
		double expectedValue = super.loadDoubleArrayFromFile("lp"+File.separator+"value"+problemId+".txt")[0];
		
		//the unbounded bounds are saved on the files with NaN values, so substitute them with acceptable values
		lb = Utils.replaceValues(lb, Double.NaN, LPPrimalDualMethod.DEFAULT_MIN_LOWER_BOUND);
		ub = Utils.replaceValues(ub, Double.NaN, LPPrimalDualMethod.DEFAULT_MAX_UPPER_BOUND);
		
		LPOptimizationRequest or = new LPOptimizationRequest();
		or.setC(c);
		or.setG(G);
		or.setH(h);
		or.setA(A);
		or.setB(b);
		or.setLb(lb);
		or.setUb(ub);
		or.setCheckKKTSolutionAccuracy(true);
//		or.setToleranceKKT(1.e-7);
//		or.setToleranceFeas(1.E-7);
//		or.setTolerance(1.E-7);
		or.setDumpProblem(true);
		//or.setPresolvingDisabled(true);
		//or.setRescalingDisabled(true);
		//or.setObjectiveFunctionDisabled(true);
		
		//optimization
		LPPrimalDualMethod opt = new LPPrimalDualMethod();
		opt.setLPOptimizationRequest(or);
		opt.optimize();
		
		LPOptimizationResponse response = opt.getLPOptimizationResponse();
		double[] sol = response.getSolution();
		RealVector cVector = new ArrayRealVector(c);
		RealVector solVector = new ArrayRealVector(sol);
		double value = cVector.dotProduct(solVector);
		logger.debug("sol   : " + ArrayUtils.toString(sol));
		logger.debug("value : "	+ value);
		
		//check constraints
		assertEquals(lb.length, sol.length);
		assertEquals(ub.length, sol.length);
		RealVector x = MatrixUtils.createRealVector(sol);
		RealMatrix GMatrix = MatrixUtils.createRealMatrix(G); 
		RealVector hvector = MatrixUtils.createRealVector(h);
		RealMatrix AMatrix = MatrixUtils.createRealMatrix(A); 
		RealVector bvector = MatrixUtils.createRealVector(b);
		for(int i=0; i<lb.length; i++){
			assertTrue(lb[i] <= x.getEntry(i));
		}
		for(int i=0; i<ub.length; i++){
			double di = Double.isNaN(lb[i])? -Double.MAX_VALUE : lb[i];
			assertTrue(di <= x.getEntry(i));
		}
		RealVector Gxh = GMatrix.operate(x).subtract(hvector);
		for(int i=0; i<Gxh.getDimension(); i++){
			double di = Double.isNaN(ub[i])? Double.MAX_VALUE : ub[i];
			assertTrue(di>= x.getEntry(i));
		}
		RealVector Axb = AMatrix.operate(x).subtract(bvector);
		assertEquals(0., Axb.getNorm(), or.getToleranceFeas());
		
		assertEquals( expectedSol.length, sol.length);
//		for(int i=0; i<sol.length; i++){
//			assertEquals(expectedSol[0], sol[0], or.getTolerance());
//		}
		//assertEquals(expectedvalue, value, or.getTolerance());
		assertTrue(Math.abs(expectedValue - value) / expectedValue < or.getTolerance());
	
	}
	
	/**
	 * Problem in the form
	 * min(c.x) s.t.
	 * G.x < h
	 * A.x = b
	 * lb <= x <= ub
	 * 
	 */
	public void xxxtestCGhAbLbUb11() throws Exception {
		logger.debug("testCGhAbLbUb11");
		
		String problemId = "11";
		
		logger.debug("problemId: " + problemId);
		double[] c = super.loadDoubleArrayFromFile("lp"+File.separator+"c"+problemId+".txt");
		double[][] G = super.loadDoubleMatrixFromFile("lp"+File.separator+"G"+problemId+".csv", ",".charAt(0));
		double[] h = super.loadDoubleArrayFromFile("lp"+File.separator+"h"+problemId+".txt");;
		double[][] A = super.loadDoubleMatrixFromFile("lp"+File.separator+"A"+problemId+".csv", ",".charAt(0));
		double[] b = super.loadDoubleArrayFromFile("lp"+File.separator+"b"+problemId+".txt");
		double[] lb = super.loadDoubleArrayFromFile("lp"+File.separator+"lb"+problemId+".txt");
		double[] ub = super.loadDoubleArrayFromFile("lp"+File.separator+"ub"+problemId+".txt");
		double[] expectedSol = super.loadDoubleArrayFromFile("lp"+File.separator+"sol"+problemId+".txt");
		double expectedValue = super.loadDoubleArrayFromFile("lp"+File.separator+"value"+problemId+".txt")[0];
		
		//the unbounded bounds are saved on the files with NaN values, so substitute them with acceptable values
		lb = Utils.replaceValues(lb, Double.NaN, LPPrimalDualMethod.DEFAULT_MIN_LOWER_BOUND);
		ub = Utils.replaceValues(ub, Double.NaN, LPPrimalDualMethod.DEFAULT_MAX_UPPER_BOUND);
		
		//check expected sol
		RealVector expectedX = MatrixUtils.createRealVector(expectedSol);
		RealMatrix GMatrix = MatrixUtils.createRealMatrix(G); 
		RealVector hvector = MatrixUtils.createRealVector(h);
		RealMatrix AMatrix = MatrixUtils.createRealMatrix(A); 
		RealVector bvector = MatrixUtils.createRealVector(b);
		for(int i=0; i<lb.length; i++){
			assertTrue(lb[i] <= expectedX.getEntry(i));
		}
		RealVector Gxh = GMatrix.operate(expectedX).subtract(hvector);
		int maxGxhIndex = super.getMaxIndex(Gxh);
		logger.debug("max Gxh pos: "+maxGxhIndex);
		logger.debug("max Gxh val: "+Gxh.getEntry(maxGxhIndex));
		for(int i=0; i<Gxh.getDimension(); i++){
			assertTrue(Gxh.getEntry(i) <= 0);
		}
		RealVector Axb = AMatrix.operate(expectedX).subtract(bvector);
		assertEquals(0., Axb.getNorm(), 1.E-6);
		
		LPOptimizationRequest or = new LPOptimizationRequest();
		or.setC(c);
		or.setG(G);
		or.setH(h);
		or.setA(A);
		or.setB(b);
		or.setLb(lb);
		or.setUb(ub);
		or.setCheckKKTSolutionAccuracy(true);
		or.setDumpProblem(true);
		//or.setPresolvingDisabled(true);
		//or.setRescalingDisabled(true);
		
		//optimization
		LPPrimalDualMethod opt = new LPPrimalDualMethod();
		opt.setLPOptimizationRequest(or);
		opt.optimize();
		
		LPOptimizationResponse response = opt.getLPOptimizationResponse();
		double[] sol = response.getSolution();
		RealVector cVector = new ArrayRealVector(c);
		RealVector solVector = new ArrayRealVector(sol);
		double value = cVector.dotProduct(solVector);
		logger.debug("sol   : " + ArrayUtils.toString(sol));
		logger.debug("value         : "	+ value);
		logger.debug("expectedValue : "	+ expectedValue);
		
		assertEquals(expectedValue, value, or.getTolerance());
	}
	
	/**
	 * Problem in the form
	 * min(c.x) s.t.
	 * G.x < h
	 * A.x = b
	 * lb <= x <= ub
	 * 
	 * Submitted by Matt MCPeak 29/06/2016.
	 */
	public void testCGhAbLbUb12() throws Exception {
		logger.debug("testCGhAbLbUb12");
		
		String problemId = "12";
		
		logger.debug("problemId: " + problemId);
		double[] c = super.loadDoubleArrayFromFile("lp"+File.separator+"c"+problemId+".txt");
		double[][] G = super.loadDoubleMatrixFromFile("lp"+File.separator+"G"+problemId+".csv", ",".charAt(0));
		double[] h = super.loadDoubleArrayFromFile("lp"+File.separator+"h"+problemId+".txt");;
		double[][] A = super.loadDoubleMatrixFromFile("lp"+File.separator+"A"+problemId+".csv", ",".charAt(0));
		double[] b = super.loadDoubleArrayFromFile("lp"+File.separator+"b"+problemId+".txt");
		double[] lb = super.loadDoubleArrayFromFile("lp"+File.separator+"lb"+problemId+".txt");
		double[] ub = super.loadDoubleArrayFromFile("lp"+File.separator+"ub"+problemId+".txt");
		double[] expectedSol = super.loadDoubleArrayFromFile("lp"+File.separator+"sol"+problemId+".txt");
		double expectedValue = super.loadDoubleArrayFromFile("lp"+File.separator+"value"+problemId+".txt")[0];
		
		//the unbounded bounds are saved on the files with NaN values, so substitute them with acceptable values
		lb = Utils.replaceValues(lb, Double.NaN, LPPrimalDualMethod.DEFAULT_MIN_LOWER_BOUND);
		ub = Utils.replaceValues(ub, Double.NaN, LPPrimalDualMethod.DEFAULT_MAX_UPPER_BOUND);
		
		//check expected sol
		RealVector expectedX = MatrixUtils.createRealVector(expectedSol);
		RealMatrix GMatrix = MatrixUtils.createRealMatrix(G); 
		RealVector hvector = MatrixUtils.createRealVector(h);
		RealMatrix AMatrix = MatrixUtils.createRealMatrix(A); 
		RealVector bvector = MatrixUtils.createRealVector(b);
		for(int i=0; i<lb.length; i++){
			assertTrue(lb[i] <= expectedX.getEntry(i));
		}
		RealVector Gxh = GMatrix.operate(expectedX).subtract(hvector);
		for(int i=0; i<Gxh.getDimension(); i++){
			assertTrue(Gxh.getEntry(i) <= 0);
		}
		RealVector Axb = AMatrix.operate(expectedX).subtract(bvector);
		assertEquals(0., Axb.getNorm(), 1.E-6);
		
		LPOptimizationRequest or = new LPOptimizationRequest();
		
		or.setC(c);
		or.setG(G);
		or.setH(h);
		or.setA(A);
		or.setB(b);
		or.setLb(lb);
		or.setUb(ub);
		or.setCheckKKTSolutionAccuracy(true);
		or.setDumpProblem(true);
		//or.setToleranceFeas(1.E-5);
		//or.setTolerance(1.E-4);
		//or.setPresolvingDisabled(true);
		//or.setRescalingDisabled(true);
		//or.setObjectiveFunctionDisabled(true);
		
		//optimization
		LPPrimalDualMethod opt = new LPPrimalDualMethod();
		opt.setLPOptimizationRequest(or);
		opt.optimize();
		
		LPOptimizationResponse response = opt.getLPOptimizationResponse();
		double[] sol = response.getSolution();
		RealVector cVector = new ArrayRealVector(c);
		RealVector solVector = new ArrayRealVector(sol);
		double value = cVector.dotProduct(solVector);
		logger.debug("sol   : " + ArrayUtils.toString(sol));
		logger.debug("value         : "	+ value);
		logger.debug("expectedValue : "	+ expectedValue);
		
		assertTrue(Math.abs(expectedValue - value) / expectedValue < or.getTolerance());
	}
	
}
