/*
 * Copyright 2025-present Solver4J
 *
 * This work is licensed under the Creative Commons Attribution-NoDerivatives 4.0 
 * International License. To view a copy of this license, visit 
 *
 *        http://creativecommons.org/licenses/by-nd/4.0/ 
 *
 * or send a letter to Creative Commons, PO Box 1866, Mountain View, CA 94042, USA.
 */
package com.solver4j.solvers;

import java.io.File;

import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.math3.linear.MatrixUtils;
import org.apache.commons.math3.linear.RealMatrix;
import org.apache.commons.math3.linear.RealVector;
import org.apache.commons.math3.linear.SingularValueDecomposition;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.solver4j.exception.SolverException;
import com.solver4j.util.Solver4JBaseTest;

/**
 * LP presolving test.
 * 
 * @author <a href="mailto:orion.waverly@gmail.com">Orion Waverly</a>
 */
public class LPPresolverTest extends Solver4JBaseTest {
	private Logger logger = LoggerFactory.getLogger(this.getClass().getName());
	
	/**
	 * Test singleton - doubleton.
	 */
	public void test1() throws Exception {
		logger.debug("test1");
		double[] c = new double[] { 1, 1, 1 };
		double[][] A = { 
				{ 1, 1, 1 }, 
				{ 3, 2, 0 } };
		double[] b = new double[] { 2, 9};
		double[] lb = new double[] { -10, -10, -10 };
		double[] ub = new double[] {  10,  10,  10 };
		double s = 0;
		double[] expectedSolution = new double[] {3, 0, -1};//this is not unique (for example {9.6666666667,-10,2.33333333})
		double expectedValue = 2;
		double expectedTolerance = 0.000000001;
		doPresolving(c, A, b, lb, ub, s, expectedSolution, expectedValue, expectedTolerance);
	}
	
	/**
	 * Test with an unconstrained variable.
	 */
	public void test2() throws Exception {
		logger.debug("test2");
		double[] c = new double[] { 1, 1, 1, 1 };
		double[][] A = { 
				{ 1, 0, 1, 1 }, 
				{ 3, 0, 2, 0 } };
		double[] b = new double[] { 2, 9};
		double[] lb = new double[] { -10, -10, -10, -10 };
		double[] ub = new double[] {  10,  10,  10,  10 };
		double s = 0;
		double[] expectedSolution = new double[] {3.0, -10.0, 0.0, -1.0};
		double expectedValue = -8;
		double expectedTolerance = 0.000000001;
		doPresolving(c, A, b, lb, ub, s, expectedSolution, expectedValue, expectedTolerance);
	}
	
	/**
	 * Test with an unconstrained variable.
	 */
	public void test3() throws Exception {
		logger.debug("test3");
		double[] c = new double[] { 1, 1, 0.333333, 1 };
		double[][] A = { 
				{ 0, 0, 0.333333, 1 }, 
				{ 0, 0, 0,        0 } };
		double[] b = new double[] { -1, 0 };
		double[] lb = new double[] { -10, -10, -10, -10 };
		double[] ub = new double[] {  10, -10,  10,  10 };
		double s = 0;
		double[] expectedSolution = new double[] {-10, -10, 0, -1};
		double expectedValue = -21;
		double expectedTolerance = 0.000000001;
		doPresolving(c, A, b, lb, ub, s, expectedSolution, expectedValue, expectedTolerance);
	}
	
	/**
	 * Test deterministic problem.
	 */
	public void test4() throws Exception {
		logger.debug("test4");
		double[] c = new double[] { 0.0, 1.0, 1.0, 1.0, 1.0, 1.0};
		double[][] A = {{1.0,0.0,-1.0,-1.0,-1.0,-1.0},{0.0,1.0,1.0,1.0,1.0,1.0},{0.0,0.0,1.0,1.0,1.0,1.0},{0.0,1.0,0.0,0.0,0.0,0.0}};
		double[] b = new double[] { -0.20127217432317202, 1.0, 0.23481753671036731, 0.7651824632896327 };
		double[] lb = new double[] { 0.0, 0.7651824632896327, 0.0, 0.0, 0.0, 0.0 };
		double[] ub = new double[] { 999999, 0.7651824632896327, 0.05870438417759183, 0.05870438417759183, 0.05870438417759183, 0.05870438417759183 };
		double s = 1;
		double[] expectedSolution = new double[] {0.0335453623871953, 0.7651824632896327, 0.05870438417759183, 0.05870438417759183, 0.05870438417759183, 0.05870438417759183};
		double expectedValue = 1d;
		double expectedTolerance = 0.000000001;
		doPresolving(c, A, b, lb, ub, s, expectedSolution, expectedValue, expectedTolerance);
	}

	public void testFromFile1() throws Exception {
		logger.debug("testFromFile1");
		
		String problemId = "1";
		
		logger.debug("problemId: " + problemId);
		double[] c = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"c"+problemId+".txt");
		double[][] A = super.loadDoubleMatrixFromFile("lp"+File.separator+"presolving"+File.separator+"A"+problemId+".csv", ",".charAt(0));
		double[] b = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"b"+problemId+".txt");
		double[] lb = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"lb"+problemId+".txt");
		double[] ub = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"ub"+problemId+".txt");
		double s = 0;
		try{
			s = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"s"+problemId+".txt")[0];
		}catch(Exception e){}
		double[] expectedSolution = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"sol"+problemId+".txt");
		double expectedValue = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"value"+problemId+".txt")[0];
		double expectedTolerance = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"tolerance"+problemId+".txt")[0];
		
		expectedTolerance = Math.max(expectedTolerance, MatrixUtils.createRealMatrix(A).operate(MatrixUtils.createRealVector(expectedSolution)).subtract(MatrixUtils.createRealVector(b)).getNorm()); 
		doPresolving(c, A, b, lb, ub, s, expectedSolution, expectedValue, expectedTolerance);
	}
	
	/**
	 * This problem has a deterministic solution.
	 */
	public void testFromFile2() throws Exception {
		logger.debug("testFromFile2");
		
		String problemId = "2";
		
		double[] c = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"c"+problemId+".txt");
		double[][] A = super.loadDoubleMatrixFromFile("lp"+File.separator+"presolving"+File.separator+"A"+problemId+".csv", ",".charAt(0));
		double[] b = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"b"+problemId+".txt");
		double[] lb = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"lb"+problemId+".txt");
		double[] ub = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"ub"+problemId+".txt");
		double s = 0;
		try{
			s = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"s"+problemId+".txt")[0];
		}catch(Exception e){}
		double[] expectedSolution = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"sol"+problemId+".txt");
		double expectedValue = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"value"+problemId+".txt")[0];
		double expectedTolerance = MatrixUtils.createRealMatrix(A).operate(MatrixUtils.createRealVector(expectedSolution)).subtract(MatrixUtils.createRealVector(b)).getNorm(); 
		
		//must be: A pXn with rank(A)=p < n
		RealMatrix AMatrix = MatrixUtils.createRealMatrix(A);
		SingularValueDecomposition dec = new SingularValueDecomposition(AMatrix);
		int rankA = dec.getRank();
		logger.debug("p: " + AMatrix.getRowDimension());
		logger.debug("n: " + AMatrix.getColumnDimension());
		logger.debug("rank: " + rankA);
		
		LPPresolver lpPresolver = new LPPresolver();
		lpPresolver.setNOfSlackVariables((short)s);
		lpPresolver.setExpectedSolution(expectedSolution);//this is just for test!
		lpPresolver.presolve(c, A, b, lb, ub);
		int n = lpPresolver.getPresolvedN();

		//deterministic solution
		assertEquals(0, n);
		assertTrue(lpPresolver.getPresolvedC() == null);
		assertTrue(lpPresolver.getPresolvedA() == null);
		assertTrue(lpPresolver.getPresolvedB() == null);
		assertTrue(lpPresolver.getPresolvedLB() == null);
		assertTrue(lpPresolver.getPresolvedUB() == null);
		assertTrue(lpPresolver.getPresolvedYlb() == null);
		assertTrue(lpPresolver.getPresolvedYub() == null);
		assertTrue(lpPresolver.getPresolvedZlb() == null);
		assertTrue(lpPresolver.getPresolvedZub() == null);
		double[] sol = lpPresolver.postsolve(new double[]{});
		assertEquals(expectedSolution.length, sol.length);
		for(int i=0; i<sol.length; i++){
			//logger.debug("i: " + i);
			assertEquals(expectedSolution[i], sol[i], 1.e-9);
		}
	}
	
	/**
	 * This problem exhibits a fixed variables that is found 
	 * during presolving after all the others simplifications.
	 * The presolver must be able to further simplify the problem
	 * after it detects this fixed variables.
	 * In altre parole, dopo aver trovato la fixed variable bisogna ricontrollare
	 * tutte le semplificazioni precedenti per vedere se si semplificano ulteriormente.   
	 * 
	 * @TODO implement this test
	 */
	public void testFromFile3() throws Exception {
		logger.debug("testFromFile3");
		
		String problemId = "3";
		
		double[] c = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"c"+problemId+".txt");
		double[][] A = super.loadDoubleMatrixFromFile("lp"+File.separator+"presolving"+File.separator+"A"+problemId+".csv", ",".charAt(0));
		double[] b = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"b"+problemId+".txt");
		
		//....
	}
	
	public void testFromFile4() throws Exception {
		logger.debug("testFromFile4");
		
		String problemId = "4";
		
		logger.debug("problemId: " + problemId);
		double[] c = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"c"+problemId+".txt");
		double[][] A = super.loadDoubleMatrixFromFile("lp"+File.separator+"presolving"+File.separator+"A"+problemId+".csv", ",".charAt(0));
		double[] b = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"b"+problemId+".txt");
		double[] lb = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"lb"+problemId+".txt");
		double[] ub = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"ub"+problemId+".txt");
		double s = 0;
		try{
			s = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"s"+problemId+".txt")[0];
		}catch(Exception e){}
		double[] expectedSolution = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"sol"+problemId+".txt");
		double expectedValue = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"value"+problemId+".txt")[0];
		double expectedTolerance = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"tolerance"+problemId+".txt")[0];
		
		expectedTolerance = Math.max(expectedTolerance, MatrixUtils.createRealMatrix(A).operate(MatrixUtils.createRealVector(expectedSolution)).subtract(MatrixUtils.createRealVector(b)).getNorm()); 
		doPresolving(c, A, b, lb, ub, s, expectedSolution, expectedValue,
				expectedTolerance);
	}
	
	/**
	 * This test involves duplicated columns.
	 */
	public void testFromFile5() throws Exception {
		logger.debug("testFromFile5");
		
		String problemId = "5";
		
		logger.debug("problemId: " + problemId);
		double[] c = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"c"+problemId+".txt");
		double[][] A = super.loadDoubleMatrixFromFile("lp"+File.separator+"presolving"+File.separator+"A"+problemId+".csv", ",".charAt(0));
		double[] b = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"b"+problemId+".txt");
		double[] lb = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"lb"+problemId+".txt");
		double[] ub = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"ub"+problemId+".txt");
		double s = 0;
		try{
			s = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"s"+problemId+".txt")[0];
		}catch(Exception e){}
		double[] expectedSolution = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"sol"+problemId+".txt");
		double expectedValue = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"value"+problemId+".txt")[0];
		double expectedTolerance = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"tolerance"+problemId+".txt")[0];
		
		expectedTolerance = Math.max(expectedTolerance, MatrixUtils.createRealMatrix(A).operate(MatrixUtils.createRealVector(expectedSolution)).subtract(MatrixUtils.createRealVector(b)).getNorm()); 
		doPresolving(c, A, b, lb, ub, s, expectedSolution, expectedValue,
				expectedTolerance);
	}
	
	public void testFromFile8() throws Exception {
		logger.debug("testFromFile8");
		
		String problemId = "8";
		
		logger.debug("problemId: " + problemId);
		double[] c = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"c"+problemId+".txt");
		double[][] A = super.loadDoubleMatrixFromFile("lp"+File.separator+"presolving"+File.separator+"A"+problemId+".csv", ",".charAt(0));
		double[] b = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"b"+problemId+".txt");
		double[] lb = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"lb"+problemId+".txt");
		double[] ub = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"ub"+problemId+".txt");
		double s = 0;
		try{
			s = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"s"+problemId+".txt")[0];
		}catch(Exception e){}
		double[] expectedSolution = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"sol"+problemId+".txt");
		double expectedValue = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"value"+problemId+".txt")[0];
		double expectedTolerance = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"tolerance"+problemId+".txt")[0];
		
		expectedTolerance = Math.max(expectedTolerance, MatrixUtils.createRealMatrix(A).operate(MatrixUtils.createRealVector(expectedSolution)).subtract(MatrixUtils.createRealVector(b)).getNorm());
		expectedTolerance = 0.0005;
		doPresolving(c, A, b, lb, ub, s, expectedSolution, expectedValue,
				expectedTolerance);
	}
	
	/**
	 * This is the afiro netlib problem presolved with Solver4J without compareBounds.
	 */
	public void testFromFile10() throws Exception {
		logger.debug("testFromFile10");
		
		String problemId = "10";
		
		logger.debug("problemId: " + problemId);
		double[] c = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"c"+problemId+".txt");
		double[][] A = super.loadDoubleMatrixFromFile("lp"+File.separator+"presolving"+File.separator+"A"+problemId+".csv", ",".charAt(0));
		double[] b = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"b"+problemId+".txt");
		double[] lb = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"lb"+problemId+".txt");
		double[] ub = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"ub"+problemId+".txt");
		double s = 0;
		try{
			s = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"s"+problemId+".txt")[0];
		}catch(Exception e){}
		double[] expectedSolution = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"sol"+problemId+".txt");
		double expectedValue = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"value"+problemId+".txt")[0];
		double expectedTolerance = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"tolerance"+problemId+".txt")[0];
		
		expectedTolerance = Math.max(expectedTolerance, MatrixUtils.createRealMatrix(A).operate(MatrixUtils.createRealVector(expectedSolution)).subtract(MatrixUtils.createRealVector(b)).getNorm()); 
		doPresolving(c, A, b, lb, ub, s, expectedSolution, expectedValue,
				expectedTolerance);
	}
	
	/**
	 * This is the presolved (with CPlex) Recipe netlib problem in standard form.
	 */
	public void testFromFile11() throws Exception {
		logger.debug("testFromFile11");
		
		String problemId = "11";
		
		logger.debug("problemId: " + problemId);
		double[] c = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"c"+problemId+".txt");
		double[][] A = super.loadDoubleMatrixFromFile("lp"+File.separator+"presolving"+File.separator+"A"+problemId+".csv", ",".charAt(0));
		double[] b = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"b"+problemId+".txt");
		double[] lb = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"lb"+problemId+".txt");
		double[] ub = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"ub"+problemId+".txt");
		double s = 0;
		try{
			s = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"s"+problemId+".txt")[0];
		}catch(Exception e){}
		double[] expectedSolution = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"sol"+problemId+".txt");
		double expectedValue = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"value"+problemId+".txt")[0];
		double expectedTolerance = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"tolerance"+problemId+".txt")[0];
		
		expectedTolerance = Math.max(expectedTolerance, MatrixUtils.createRealMatrix(A).operate(MatrixUtils.createRealVector(expectedSolution)).subtract(MatrixUtils.createRealVector(b)).getNorm());
		expectedTolerance = 1.e-9;
		doPresolving(c, A, b, lb, ub, s, expectedSolution, expectedValue,
				expectedTolerance);
	}
	
	/**
	 * This is the VTP.BASE netlib problem in standard form.
	 */
	public void testFromFile12() throws Exception {
		logger.debug("testFromFile12");
		
		String problemId = "12";
		
		logger.debug("problemId: " + problemId);
		double[] c = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"c"+problemId+".txt");
		double[][] A = super.loadDoubleMatrixFromFile("lp"+File.separator+"presolving"+File.separator+"A"+problemId+".csv", ",".charAt(0));
		double[] b = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"b"+problemId+".txt");
		double[] lb = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"lb"+problemId+".txt");
		double[] ub = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"ub"+problemId+".txt");
		double s = 0;
		try{
			s = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"s"+problemId+".txt")[0];
		}catch(Exception e){}
		double[] expectedSolution = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"sol"+problemId+".txt");
		double expectedValue = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"value"+problemId+".txt")[0];
		double expectedTolerance = super.loadDoubleArrayFromFile("lp"+File.separator+"presolving"+File.separator+"tolerance"+problemId+".txt")[0];
		
		expectedTolerance = Math.max(expectedTolerance, MatrixUtils.createRealMatrix(A).operate(MatrixUtils.createRealVector(expectedSolution)).subtract(MatrixUtils.createRealVector(b)).getNorm()); 
		doPresolving(c, A, b, lb, ub, s, expectedSolution, expectedValue,
				expectedTolerance);
	}

	private void doPresolving(double[] c, double[][] A, double[] b, double[] lb,
			double[] ub, double s, double[] expectedSolution, double expectedValue,
			double expectedTolerance) throws SolverException{
		
		RealMatrix AMatrix = MatrixUtils.createRealMatrix(A);
		SingularValueDecomposition dec = new SingularValueDecomposition(AMatrix);
		int rankA = dec.getRank();
		logger.debug("p: " + AMatrix.getRowDimension());
		logger.debug("n: " + AMatrix.getColumnDimension());
		logger.debug("rank: " + rankA);
		
		LPPresolver lpPresolver = new LPPresolver();
		lpPresolver.setNOfSlackVariables((short)s);
		lpPresolver.setExpectedSolution(expectedSolution);//this is just for test!
		//lpPresolver.setExpectedTolerance(expectedTolerance);//this is just for test!
		lpPresolver.presolve(c, A, b, lb, ub);
		int n = lpPresolver.getPresolvedN();
		if(n==0){
			//deterministic solution
			assertEquals(0, n);
			assertTrue(lpPresolver.getPresolvedC() == null);
			assertTrue(lpPresolver.getPresolvedA() == null);
			assertTrue(lpPresolver.getPresolvedB() == null);
			assertTrue(lpPresolver.getPresolvedLB() == null);
			assertTrue(lpPresolver.getPresolvedUB() == null);
			assertTrue(lpPresolver.getPresolvedYlb() == null);
			assertTrue(lpPresolver.getPresolvedYub() == null);
			assertTrue(lpPresolver.getPresolvedZlb() == null);
			assertTrue(lpPresolver.getPresolvedZub() == null);
			
			//check objective function
			double delta = expectedTolerance;
			RealVector postsolvedX = MatrixUtils.createRealVector(lpPresolver.postsolve(new double[]{}));
			double value = MatrixUtils.createRealVector(c).dotProduct(postsolvedX);
			assertEquals(expectedSolution.length, postsolvedX.getDimension());
			assertEquals(expectedValue, value, delta);
			for(int i=0; i<postsolvedX.getDimension(); i++){
				//logger.debug("i: " + i);
				assertEquals(expectedSolution[i], postsolvedX.getEntry(i), 1.e-9);
			}
			
			//check postsolved constraints
			for(int i=0; i<lb.length; i++){
				double di = Double.isNaN(lb[i])? -Double.MAX_VALUE : lb[i];
				assertTrue(di <= postsolvedX.getEntry(i) + delta);
			}
			for(int i=0; i<ub.length; i++){
				double di = Double.isNaN(ub[i])? Double.MAX_VALUE : ub[i];
				assertTrue(di + delta >= postsolvedX.getEntry(i));
			}
			RealVector Axmb = AMatrix.operate(postsolvedX).subtract(MatrixUtils.createRealVector(b));
			assertEquals(0., Axmb.getNorm(), expectedTolerance);
		}else{
			double[] presolvedC = lpPresolver.getPresolvedC().toArray();
			double[][] presolvedA = lpPresolver.getPresolvedA().toArray();
			double[] presolvedB = lpPresolver.getPresolvedB().toArray();
			double[] presolvedLb = lpPresolver.getPresolvedLB().toArray();
			double[] presolvedUb = lpPresolver.getPresolvedUB().toArray();
			double[] presolvedYlb = lpPresolver.getPresolvedYlb().toArray();
			double[] presolvedYub = lpPresolver.getPresolvedYub().toArray();
			double[] presolvedZlb = lpPresolver.getPresolvedZlb().toArray();
			double[] presolvedZub = lpPresolver.getPresolvedZub().toArray();
			logger.debug("n  : " + n);
			logger.debug("presolvedC  : " + ArrayUtils.toString(presolvedC));
			logger.debug("presolvedA  : " + ArrayUtils.toString(presolvedA));
			logger.debug("presolvedB  : " + ArrayUtils.toString(presolvedB));
			logger.debug("presolvedLb : " + ArrayUtils.toString(presolvedLb));
			logger.debug("presolvedUb : " + ArrayUtils.toString(presolvedUb));
			logger.debug("presolvedYlb: " + ArrayUtils.toString(presolvedYlb));
			logger.debug("presolvedYub: " + ArrayUtils.toString(presolvedYub));
			logger.debug("presolvedZlb: " + ArrayUtils.toString(presolvedZlb));
			logger.debug("presolvedZub: " + ArrayUtils.toString(presolvedZub));
			
			//check objective function
			double delta = expectedTolerance;
			RealVector presolvedX = MatrixUtils.createRealVector(lpPresolver.presolve(expectedSolution));
			logger.debug("presolved value: " + MatrixUtils.createRealVector(presolvedC).dotProduct(presolvedX));
			RealVector postsolvedX = MatrixUtils.createRealVector(lpPresolver.postsolve(presolvedX.toArray()));
			double value = MatrixUtils.createRealVector(c).dotProduct(postsolvedX);
			assertEquals(expectedValue, value, delta);
			
			//check postsolved constraints
			for(int i=0; i<lb.length; i++){
				double di = Double.isNaN(lb[i])? -Double.MAX_VALUE : lb[i];
				assertTrue(di <= postsolvedX.getEntry(i) + delta);
			}
			for(int i=0; i<ub.length; i++){
				double di = Double.isNaN(ub[i])? Double.MAX_VALUE : ub[i];
				assertTrue(di + delta >= postsolvedX.getEntry(i));
			}
			RealVector Axmb = AMatrix.operate(postsolvedX).subtract(MatrixUtils.createRealVector(b));
			assertEquals(0., Axmb.getNorm(), expectedTolerance);
			
			//check presolved constraints
			assertEquals(presolvedLb.length, presolvedX.getDimension());
			assertEquals(presolvedUb.length, presolvedX.getDimension());
			AMatrix = MatrixUtils.createRealMatrix(presolvedA); 
			RealVector bvector = MatrixUtils.createRealVector(presolvedB);
			for(int i=0; i<presolvedLb.length; i++){
				double di = Double.isNaN(presolvedLb[i])? -Double.MAX_VALUE : presolvedLb[i];
				assertTrue(di <= presolvedX.getEntry(i) + delta);
			}
			for(int i=0; i<presolvedUb.length; i++){
				double di = Double.isNaN(presolvedUb[i])? Double.MAX_VALUE : presolvedUb[i];
				assertTrue(di + delta >= presolvedX.getEntry(i));
			}
			Axmb = AMatrix.operate(presolvedX).subtract(bvector);
			assertEquals(0., Axmb.getNorm(), expectedTolerance);
			
			//check rank(A): must be A pXn with rank(A)=p < n
			AMatrix = MatrixUtils.createRealMatrix(presolvedA);
			dec = new SingularValueDecomposition(AMatrix);
			rankA = dec.getRank();
			logger.debug("p: " + AMatrix.getRowDimension());
			logger.debug("n: " + AMatrix.getColumnDimension());
			logger.debug("rank: " + rankA);
			assertEquals(AMatrix.getRowDimension(), rankA);
			assertTrue(rankA < AMatrix.getColumnDimension());
		}
	}
}
