package com.solver4j;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.regex.Pattern;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.github.classgraph.ClassGraph;
import io.github.classgraph.ClassInfo;
import io.github.classgraph.ScanResult;
import junit.framework.TestCase;

public class CircularDependencyTest extends TestCase {
	
	private String basePackage = CircularDependencyTest.class.getPackage().getName(); // root package
	private Pattern testClassPattern = Pattern.compile(".*(Test|Tests|TestCase)$");
	private Logger logger = LoggerFactory.getLogger(this.getClass().getName());

	public void testNoCircularPackageDependencies() throws Exception {
        try (ScanResult scanResult = new ClassGraph()
                .acceptPackages(CircularDependencyTest.class.getPackage().getName())
                .enableClassInfo()
                .enableInterClassDependencies()
                .scan()) {

            Map<String, Set<String>> dependencies = new HashMap<>();

            for (ClassInfo ci : scanResult.getAllClasses()) {
                String fromPackage = ci.getPackageName();
                ci.getClassDependencies().forEach(dep -> {
                    String toPackage = dep.getPackageName();
                    if (!fromPackage.equals(toPackage)) {
                        dependencies
                            .computeIfAbsent(fromPackage, k -> new HashSet<>())
                            .add(toPackage);
                    }
                });
            }

            // Detect cycles
            for (String pkg : dependencies.keySet()) {
                if (hasCycle(pkg, dependencies)) {
                    fail("Circular dependency detected starting at package: " + pkg);
                }
            }
        }
    }
	
	/**
	 * Does not work properly.
	 * @throws Exception
	 */
	public void xxxtestNoCircularClassDependencies() throws Exception {	
        Set<String> testBaseClassNames = new HashSet<>();
        try (ScanResult scanResult = new ClassGraph()
                .acceptPackages(basePackage)
                .enableClassInfo()
                .enableInterClassDependencies()
                .ignoreClassVisibility()
                .scan()) {

            // First pass: identify all test classes
            for (ClassInfo classInfo : scanResult.getAllClasses()) {
                String className = classInfo.getName();
                if (isTestClass(className, testClassPattern)) {
                    testBaseClassNames.add(stripInnerClass(className));
                }
            }

            Map<String, Set<String>> dependencyGraph = new HashMap<>();

            // Second pass: build dependency graph excluding test & inner test classes
            for (ClassInfo classInfo : scanResult.getAllClasses()) {
                String className = classInfo.getName();

                if (isOrIsInnerOfTest(className, testBaseClassNames)) continue;
                if (className.equals("com.solver4j.optimizers.NewtonUnconstrained")) continue;

                Set<String> deps = new HashSet<>();
                classInfo.getClassDependencies().forEach(dep -> {
                    String depName = dep.getName();

                    if (className.equals(depName)) return; // self
                    if (stripInnerClass(className).equals(stripInnerClass(depName))) return; // inner/outer
                    if (isOrIsInnerOfTest(depName, testBaseClassNames)) return; // test or inner of test

                    logger.debug(className + " depends on " + depName);
                    deps.add(depName);
                });

                dependencyGraph.put(className, deps);
            }

            // detect cycles
            for (String clazz : dependencyGraph.keySet()) {
                if (hasCycle(clazz, dependencyGraph)) {
                    fail("Circular dependency detected starting at class: " + clazz);
                }
            }
        }
    }

	private boolean hasCycle(String current, Map<String, Set<String>> deps) {
		Set<String> dependencies = deps.get(current);
		for (String dep : dependencies) {
			if (deps.get(dep) != null && deps.get(dep).contains(current)) {
				logger.error("Circular dependency found: " + current + " depends on " + dep + " while " + dep + " depends on " + current);
				return true;
			}
		}
		return false;
	}
    
    private String stripInnerClass(String className) {
        return className.split("\\$")[0];
    }
    
    private boolean isTestClass(String className, Pattern testPattern) {
        return className.contains(".test.") ||
               className.contains(".tests.") ||
               testPattern.matcher(className).matches();
    }
    
    private boolean isOrIsInnerOfTest(String className, Set<String> testBaseClasses) {
        String base = stripInnerClass(className);
        return testBaseClasses.contains(base);
    }
}
